/* $Id: readpemler.c,v 2.4 2003/08/23 22:25:55 friedrich Exp $
  this file contains the subroutines
     readpemler1
     readpemler1_
     readpemler2
     readpemler2_
     readpemler
  which read one and two dimensional spectra, respectively
*/

#include <stdlib.h>
/*
 This routine reads one dimensional spectra in the binary format
   defined by Peter Pemler   LMU Muenchen

   The file has the following structure (Stand Nov. 1993):

   long id:              4 Byte Identifications code
   char name[16]         16 Byte Spectrum Name
   long depth            4 Byte Kanalbreite in Bytes
   long activ            nur fuer Replay system
   long slength          Spektrumlaenge in Byte
   specdim dim[1]        definiert die Eigenschaften der Dimensionen

   mit
   specdim = struct
   { 
     char evtcomp[8]     Namen der dimension
     long min            untere Grenze
     long max            obere Grenze
     long compress       Komprimierungsfaktor
     long length         Laenge der Dimension in Kanaelen
   }


  int readpemler(
     char *filename:  Filename of Spectrum. If Filename ends with .Z, it
                      will be assumed, that the file is compressed. 
                      An uncompressed file is generated under /usr/tmp
                      and used for reading. This file will be deleted
                      with a subsequent calls with an other filename. 
    int subspectrum:  Number of the subspectrum.
    double *spectrum:  Array of the spectrum. Must have length of at least
                      ende - begin + 1
    int begin:        begin of the Spectrum
    int ende:         end of the spectrum
    )
  return value:
    0 if the file is not found or if there was an error during reading
   <0 negativ value of last subspectrum if subspectrum does not exist
   >0 ende or last channel in the spectrum if ende is greater than 
      the last channel
  
   F. Riess, Nov. 1993
*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <unistd.h>

typedef struct { char evtcomp[8];
                 long min, max, compress, length;
               } specdim;
typedef struct { char name[16];
	 	 long depth, active, slength;
                 specdim dim[1];
	       } PmSpecd1;
typedef struct { char name[16];
	 	 long depth, active, slength;
                 specdim dim[2];
	       } PmSpecd2;
typedef double SPECTYPE;

int header_out;
int readpemler(char *filename, int subspectrum, SPECTYPE *spectrum, int begin, int ende);
int readpemler_(char*, int*, SPECTYPE*, int*, int*, int filenamelength);


/* routine to read single spectra */

int readpemler1(char *filename, int subspectrum, SPECTYPE *spectrum, int begin, int ende)
{
  extern int header_out;
  header_out = 1;
  return readpemler(filename, subspectrum, spectrum, begin, ende);
}


/* Fortran interface */
int readpemler1_(char *fi, int *su, SPECTYPE *sp, int *be, int *en, int fl)
{
  extern int header_out;
  header_out = 1;
  return readpemler_(fi, su, sp, be, en, fl);
}

/* routine to read matrix */

int readpemler2(char *filename, int subspectrum, SPECTYPE *spectrum, int begin, int ende)
{
  extern int header_out;
  header_out = 0;
  return readpemler(filename, subspectrum, spectrum, begin, ende);
}


/* Fortran interface */
int readpemler2_(char *fi, int *su, SPECTYPE *sp, int *be, int *en, int fl)
{
  extern int header_out;
  header_out = 0;
  return readpemler_(fi, su, sp, be, en, fl);
}


int readpemler(char *filename, int subspectrum, SPECTYPE *spectrum, int begin, int ende)
{
  extern int header_out;
  int i, j, n, length_bytes;
  char str[131];
  long start, assembly;
  static char filetmp[L_tmpnam > 130 ? L_tmpnam : 130] = "";
  static char fileold[130] = " ";
  static int firstfile = 1, filesub = -1, spectrumsize;
  static FILE *fp = NULL;
  static SPECTYPE *data = NULL;
  static unsigned char *idata = NULL;
  static PmSpecd2 header;
  static long id;  
    
/* check if file has already been accessed */
  
  if(strcmp(fileold, filename))
  {
    if(fp)
    {
      fclose(fp); fp = NULL;
      if(!firstfile) (void) unlink(filetmp);
      firstfile = 1;
      filesub = -1;
    }     
    strcpy(fileold, filename);
    j = strlen(filename) - 1;
    if(!strcmp(&filename[j-1], ".Z")) j = 0;
    else if(!strcmp(&filename[j-2], ".gz")) j = -1;
    if(j > 0)
      strcpy(filetmp, filename);
    else
    { if(firstfile)
      { strcpy(filetmp, "/tmp/gsppemlerXXXXXX");
        j = mkstemp(filetmp);
        if(j < 0) return 0;
        close(j);
        firstfile = 0;
      }
      if(access(filename, 00) == -1) return 0;
      sprintf(str, "zcat %s > %s\n", filename, filetmp);
      system(str);
    }
    if((fp = fopen(filetmp, "r")) == (FILE *) 0) return 0;
/* read spectrum header */
    fread(&id, 4, 1, fp);
    if(id == 0x1d1d)
    {
      fread(&header, sizeof(PmSpecd1), 1, fp);
      header.dim[1].min = header.dim[1].max = 0;
    }
    else if(id == 0x2d2d)
      fread(&header, sizeof(PmSpecd2), 1, fp);
    else 
    {
      printf("--> readpemler: unknown spectrum ID\n");
      fclose(fp);  fp = NULL;  *fileold = '\0';
      return 0;
    }
    if(header_out)
    {
      printf(" name: %s\n Wordlength: %ld Bytes", header.name, header.depth);
      printf(" Spectrum dimension %d\n", id == 0x1d1d ? 1 : 2);
      printf(" dimension 1 from %5ld to %5ld\n",
                            header.dim[0].min, header.dim[0].max); 
      if(id == 0x2d2d)
        printf(" dimension 2 from %5ld to %5ld\n",
                          header.dim[1].min, header.dim[1].max); 
    }
    if(idata) { free(idata); idata = NULL; }
    if(data)  { free(data);  data = NULL; }
  }
/* the file is open check for the subspectrum */
/* Note: subspectra start with 0 */
  if(filesub != subspectrum)
  {
    if(subspectrum > header.dim[1].max) return -header.dim[1].max;
    if(subspectrum < header.dim[1].min) return -header.dim[1].max;
    spectrumsize = (header.dim[0].max - header.dim[0].min + 1);
    if(!data)
      if((data = (SPECTYPE *) calloc(spectrumsize, sizeof(SPECTYPE))) == NULL)
      {
        fclose(fp);  fp = NULL;  *fileold = '\0';
        return -1;
      }
    length_bytes = spectrumsize * header.depth;
    start = (long) sizeof(id) + (subspectrum - header.dim[1].min) * length_bytes;
    if(id == 0x1d1d) start += sizeof(PmSpecd1);
    else             start += sizeof(PmSpecd2);
    if(start > 0)
    {
      if(fseek(fp, start, SEEK_SET))
      {
        fclose(fp);  fp = NULL;  *fileold = '\0';
        return -1;
      }
      if(!idata)
        if((idata = (unsigned char *) malloc(length_bytes)) == NULL)
        {
          fclose(fp); fp = NULL;  *fileold = '\0';
          return -1;
        }
      n = fread(idata, 1, length_bytes, fp);
      if(n != length_bytes)
      {
        fclose(fp); fp = NULL;  *fileold = '\0';
	return 0;
      }
/* assemble bytes into a word and convert it to double  */          
      for(i = j = 0; j < spectrumsize; ++j )
      {
        for(n = 0, assembly = 0; n < header.depth; ++n)
        {
          assembly <<= 8;
          assembly += idata[i];
          ++i;
        }
        data[j] = (SPECTYPE) assembly;
      }
    }
    else
      for(i = 0; i < spectrumsize; ++i) data[i] = 0.;	 
    filesub = subspectrum;
    free(idata);
    idata = NULL;
  }
/* Spectrum is in buffer, transfer it to extern buffer */
  if(ende >= spectrumsize) ende = spectrumsize - 1 ;
  j = begin - header.dim[0].min;
  for(i = 0; j <= ende; ++i, ++j)
    if(j < 0) spectrum[i] = 0;
    else      spectrum[i] = data[j];
  return (ende + header.dim[0].min);
}  


/* Fortran interface */
int readpemler_(char *fi, int *su, SPECTYPE *sp, int *be, int *en, int fl)
{
  char file[130];
  int i;

  for(i = 0; i < fl; ++i)
  { if(fi[i] == ' ')
    { file[i] = '\0';
      break;
    }
    else file[i] = fi[i];
  }
  return readpemler(file, *su, sp, *be, *en);
}
