/* gspcompare

   compares 2 gsp-output with each other.
   A differences will be broadcasted if 
   1. peaks are missing in one of the files
   2. peak data differ by more than 20 % of
      the statistical error

   f.Riess, March 2000
*/

#include <stdio.h>
#include <math.h> 
#include <stdlib.h>

#define VERSION "10.04.2003"
#define PM "+/-"
#define PEAKS 1000
#define STRSIZE 132
#define LIMITP 0.20          /* deviation limit for centroid */
#define LIMITA 0.20          /* deviation limit for area */
#define LIMITW 0.20          /* deviation limit for width */
#define LIMITD 0.20          /* deviation limit for all errors */

int istext(char *str);
int next (double element, double array[], int anzahl);
char *equalexp(double xx, double dxx, int digits, char *between);

int main(int argc, char **argv)
{
  char *file1, *file2;
  FILE *fp1, *fp2;
  double peak1[PEAKS], dpeak1[PEAKS], area1[PEAKS], darea1[PEAKS], width1[PEAKS], dwidth1[PEAKS];
  double epeak1[PEAKS], depeak1[PEAKS], earea1[PEAKS], dearea1[PEAKS], ewidth1[PEAKS], dewidth1[PEAKS];
  double peak2[PEAKS], dpeak2[PEAKS], area2[PEAKS], darea2[PEAKS], width2[PEAKS], dwidth2[PEAKS];
  double epeak2[PEAKS], depeak2[PEAKS], earea2[PEAKS], dearea2[PEAKS], ewidth2[PEAKS], dewidth2[PEAKS];
  double pdiff, adiff, wdiff, epdiff, eadiff, ewdiff, diff;
  double dpdiff, dadiff, dwdiff, depdiff, deadiff, dewdiff, ddiff;
  char str[STRSIZE];
  int n1, n2, i, i1, i2, j, anzahl1 = 0, anzahl2 = 0, count, output = 0, debug = 0;
  int indx1[PEAKS], indx2[PEAKS];
  if(argc < 3)
  {
    printf("call: gspcompare file1 file2 [-g]\n");
    exit(1);
  }
  file1 = argv[1];    file2 = argv[2];
  if((fp1 = fopen(file1, "r")) == NULL)
  { 
    fprintf(stderr, "cannot open file: %s\n", file1);
    exit(1);
  }
  if((fp2 = fopen(file2, "r")) == NULL)
  { 
    fprintf(stderr, "cannot open file: %s\n", file2);
    exit(1);
  }
  if(argc == 4) debug = 1;
  printf("gspcompare (%s%s)\n", VERSION, debug ? ", debug mode on" : "");
  printf("centroids must agree within %3.0f %% of their errors\n", 100*LIMITP);
  printf("areas must agree within     %3.0f %% of their errors\n", 100*LIMITA);
  printf("widths must agree within    %3.0f %% of their errors\n", 100*LIMITW);
  printf("errors must agree within    %3.0f %%                \n", 100*LIMITD);
  n1 = n2 = 0;
  while(fgets(str, STRSIZE, fp1) != NULL)
  {
    if(istext(str)) continue;
    anzahl1 = sscanf(str, "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
         &peak1[n1], &dpeak1[n1], &area1[n1], &darea1[n1], &width1[n1], &dwidth1[n1], 
         &epeak1[n1], &depeak1[n1], &earea1[n1], &dearea1[n1], &ewidth1[n1], &dewidth1[n1]);
    if(anzahl1 == 10)
    { 
      ewidth1[n1]  = earea1[n1];
      dewidth1[n1] = dearea1[n1];
    }
    if(debug) printf("%3d %9.3f %7.3f %12.4e %12.4e %8.2f %8.2f\n",
       n1 + 1, peak1[n1], dpeak1[n1], area1[n1], darea1[n1], width1[n1], dwidth1[n1]);
    indx1[n1] = n1;
    ++n1;
  }
  fclose(fp1);
  while(fgets(str, STRSIZE, fp2) != NULL)
  {
    if(istext(str)) continue;
    anzahl2 = sscanf(str, "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
         &peak2[n2], &dpeak2[n2], &area2[n2], &darea2[n2], &width2[n2], &dwidth2[n2], 
         &epeak2[n2], &depeak2[n2], &earea2[n2], &dearea2[n2], &ewidth2[n2], &dewidth2[n2]);
    if(anzahl2 == 10)
    { 
      ewidth2[n2]  = earea2[n2];
      dewidth2[n2] = dearea2[n2];
    }
    indx2[n2] = n2;
    if(debug) printf("%3d %9.3f %7.3f %12.4e %12.4e %8.2f %8.2f\n",
       n2 + 1, peak2[n2], dpeak2[n2], area2[n2], darea2[n2], width2[n2], dwidth2[n2]);
    ++n2;
  }
  fclose(fp2);
  if(anzahl1 != anzahl2)
  {
    ++output;
    printf("Data differ in calibration mode:\n");
    printf(" %s: %s calibration\n %s: %s calibration\n",
           file1, anzahl1 == 6 ? "no" : anzahl1 == 10 ? "energy" : "energy + efficiency",
           file2, anzahl2 == 6 ? "no" : anzahl2 == 10 ? "energy" : "energy + efficiency");
  }
  else if(anzahl1 >= 10)
  {
    for(i = 0; i < n1; ++i)
    {
      peak1[i] = epeak1[i];
      dpeak1[i] = depeak1[i];
      width1[i] = ewidth1[i];
      dwidth1[i] = dewidth1[i];
      if(anzahl1 < 12) continue;
      area1[i] = earea1[i];
      darea1[i] = dearea1[i];
    }
    for(i = 0; i < n2; ++i)
    {
      peak2[i] = epeak2[i];
      dpeak2[i] = depeak2[i];
      width2[i] = ewidth2[i];
      dwidth2[i] = dewidth2[i];
      if(anzahl2 < 12) continue;
      area2[i] = earea2[i];
      darea2[i] = dearea2[i];
    }
  }
  if(n1 != n2)
  {
    ++output;
#ifdef FILENAMES
    printf("Data differ in number of peaks: %s: %d, %s: %d\n",
           file1, n1, file2, n2);
#else
    printf("Data differ in number of peaks: this fit: %d, standard: %d\n", n1, n2);
#endif
  }
  pdiff = adiff = wdiff = epdiff = eadiff = ewdiff = 0;
  dpdiff = dadiff = dwdiff = depdiff = deadiff = dewdiff = 0;
  for(i = 0; i <= 8 && n1 && n2; ++i)
  {
    count = 0;
    i1 = 0;
    pdiff = adiff = wdiff = epdiff = eadiff = ewdiff = 0;
    dpdiff = dadiff = dwdiff = depdiff = deadiff = dewdiff = 0;
    while(i1 < n1)
    {
      i2 = next(peak1[i1], peak2, n2);
      dpdiff = sqrt(dpeak1[i1]*dpeak1[i1] + dpeak2[i2]*dpeak2[i2]); 
      if(dpdiff < 0.001) dpdiff = 0.001;
      diff = pdiff = fabs(peak1[i1] - peak2[i2])/dpdiff;
      ddiff = dpdiff = 0.707*fabs(dpeak1[i1] - dpeak2[i2])/dpdiff;
      dadiff = sqrt(darea1[i1]*darea1[i1] + darea2[i2]*darea2[i2]);
      if(dadiff == 0.) dadiff = 1.;
      adiff = fabs(area1[i1] - area2[i2])/dadiff;
      dadiff = 0.707*fabs(darea1[i1] - darea2[i2])/dadiff;
      if(diff < adiff) diff = adiff;
      if(ddiff < dadiff) ddiff = dadiff;
      dwdiff = sqrt(dwidth1[i1]*dwidth1[i1] + dwidth2[i2]*dwidth2[i2]);
      if(dwdiff < 0.01) dwdiff = 0.01;
      wdiff = fabs(width1[i1] - width2[i2])/dwdiff;
      dwdiff = 0.707*fabs(dwidth1[i1] - dwidth2[i2])/dwdiff;
      if(diff < wdiff) diff = wdiff;
      if(ddiff < dwdiff) ddiff = dwdiff;
/*
   fprintf(stderr, "%3d %9.3f %7.2f %7.2f %7.2f %7.2f\n",i, peak1[i1],pdiff,adiff,wdiff,ddiff);
*/
      if(i == 0 && pdiff <= LIMITP && adiff <= LIMITA && wdiff <= LIMITW && ddiff <= LIMITD)
      {
        ++count;
        indx1[i1] = indx2[i2] = -1;
      }
      else if(i && pdiff <= 0.5*i && adiff <= 0.5*i && wdiff <= 0.5*i)
      {
        if(i && count == 0)
          printf("searching for agreement of peaks within factor %3.1f of statistical error\n", 0.5*i);
          ++output;
          ++count;
          indx1[i1] = indx2[i2] = -1;
#ifdef FILENAMES
          printf("%20s: %9.3f %9.3f %s %7.2f %7.2f\n", file1, peak1[i1], dpeak1[i1],
                 equalexp(area1[i1], darea1[i1], 2, " "), width1[i1], dwidth1[i1]);
          printf("%20s: %9.3f %9.3f %s %7.2f %7.2f\n\n", file2, peak2[i2], dpeak2[i2], 
                 equalexp(area2[i2], darea2[i2], 2, " "), width2[i2], dwidth2[i2]);
#else
          printf("  this fit: %11.3f %8.3f", peak1[i1], dpeak1[i1]);
          printf(" %s",            equalexp(area1[i1], darea1[i1], 2, PM));
          printf(" %s\n",          equalexp(width1[i1], dwidth1[i1], 2, PM));
          printf("  standard: %11.3f %8.3f", peak2[i2], dpeak2[i2]);
          printf(" %s",            equalexp(area2[i2], darea2[i2], 2, PM));
          printf(" %s\n\n",        equalexp(width2[i2], dwidth2[i2], 2, PM));
#endif
      } 
      ++i1;
    }
    if(count > 0)
      printf("%d peak%s within these limits\n", count, count == 1 ? " is" : "s are");
/* eliminate identified peaks */
    for(j = i1 = 0; i1 < n1; ++i1)
    {
      if(indx1[i1] < 0) continue;
      peak1[j] = peak1[i1];             dpeak1[j] = dpeak1[i1];
      area1[j] = area1[i1];             darea1[j] = darea1[i1];
      width1[j] = width1[i1];           dwidth1[j] = dwidth1[i1];
      indx1[j] = j;
      j = j + 1;
    }
    n1 = j;
    for(j = i2 = 0; i2 < n2; ++i2)
    {
      if(indx2[i2] < 0) continue;
      peak2[j] = peak2[i2];             dpeak2[j] = dpeak2[i2];
      area2[j] = area2[i2];             darea2[j] = darea2[i2];
      width2[j] = width2[i2];           dwidth2[j] = dwidth2[i2];
      indx2[j] = j;
      j = j + 1;
    }
    n2 = j;
  }
  printf("extra peaks\n");
  if(n1 > 0)
  {
    ++output;
#ifdef FILENAMES
    printf("%s\n", file1);
#else
    printf("  this fit:\n");
#endif
    for(i1 = 0; i1 < n1; ++i1)
        printf("%20s: %9.3f %9.3f %12.0f %8.0f %7.2f %7.2f\n",
            " ", peak1[i1], dpeak1[i1], area1[i1], darea1[i1], width1[i1], dwidth1[i1]);
  }
  if(n2 > 0)
  {
    ++output;
#ifdef FILENAMES
    printf("%s\n", file2);
#else
    printf("  standard:\n");
#endif
    for(i2 = 0; i2 < n2; ++i2)
      printf("%20s: %9.3f %9.3f %12.0f %8.0f %7.2f %7.2f\n",
            " ", peak2[i2], dpeak2[i2], area2[i2], darea2[i2], width2[i2], dwidth2[i2]);
  }
  if(output) system("touch gspcompare.bak");
  else       system("rm -f gspcompare.bak");
  return (output);
}

/*  int istext(char *text)

    routine to check if a string is a text string or a number string
    it returns 1 if a text string is idendified
               0 otherwise

    a non text string is a string consisting of numbers:
       a sequence of digits [seperated by a delimiter]:  122 23 33\n
       a sequence of digits with a dot: .122 2.3  33.\n
       a sequence of digits preceeded by a sign: +122 +.23 -0.33\n
       a sequence of digits with an e:  +.23e-00 -1.23E+03\n

    delimiters a space, tab, return or a comma if preceeded by a digit

    F. Riess August 1994
*/

#include <stdio.h>
int istext(char *str)
{
  char *pstr;
  int was_space, was_digit, was_sign, was_dot, was_e, was_komma, text_flag;
  int number, number_tot, sign, dot, e;

  was_space = 1;
  was_digit = was_sign = was_dot = was_e = was_komma = text_flag = 0;
  number = number_tot = sign = dot = e = 0;
  for(pstr = str; *pstr && !text_flag; ++pstr)
  {
    switch(*pstr)
    {
      case '0':;
      case '1':;
      case '2':;
      case '3':;
      case '4':;
      case '5':;
      case '6':;
      case '7':;
      case '8':;
      case '9': if(was_e) text_flag = 1;
                was_digit = 1;   ++number;    ++number_tot;
                was_sign = was_dot = was_e = was_komma = was_space = 0;
                break;
      case '.': if(dot || was_e) text_flag = 1;
                was_dot = 1;     ++dot;
                was_digit = was_sign = was_e = was_komma = was_space = 0;
                break;
      case '+': ;
      case '-': if(was_sign || was_digit || was_dot) text_flag = 1;
                was_sign = 1;    ++sign;
                was_digit = was_dot = was_e = was_komma = was_space = 0;
                break;
      case 'E': ;
      case 'e': if(!(was_digit || (was_dot && number))) text_flag = 1;
                was_e = 1;       ++e;
                was_digit = was_sign = was_dot = was_komma = was_space = 0;
                break;
      case ',': if(!(was_digit || (was_dot && number))) text_flag = 1;
      case '\t': ;
      case '\n': ;
      case '\r': ;
      case '\v': ;
      case '\f': ;
      case ' ': if(was_e || was_sign || e > 1 || dot > 1 || sign > 2 ||
                   (sign == 2 && !e)) text_flag = 1; 
                was_space = 1;
                was_digit = was_sign = was_dot = was_e = was_komma = 0;
                number = sign = dot = e = 0;
                was_dot = was_komma = was_e = was_sign = was_digit = 0;
                break;
      default: text_flag = 1;
    }
  }
  if(!number_tot) text_flag = 1;
  return text_flag;
}
  
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  Name:                  int next (double element, double *array, int anzahl)
  Funktion:              Die Funktion sucht in dem Array array mit der
			                   Laenge anzahl den zu element naechstgelegensten
                         Index als Funktionswert zurueck. Die Werte in
                         array muessen entweder monoton steigen oder
                          abnehmen.
  Verfahren:              Sukzessive Aproximation.
  Benoetigte Funktionen:  keine.
  Autor:                  Friedrich Riess, 27.11.86
  Aenderungen:
-----------------------------------------------------------------------*/
/* Diese Funktion sorgt dafuer, dass L <= V <= U ist.     */
#define LIMIT(L,V,U)  ((V) < (L) ? (L) : (V) <= (U) ? (V) : (U))

int next (double element, double array[], int anzahl)
{
   int signum, n = -1, rest = 2;
   anzahl--;
   signum = (array[anzahl] >= array[0])? 1: -1;
   while (rest <= anzahl) rest *= 2; rest /= 2;
   while (rest)
   {
       n += rest; n = LIMIT (0, n, anzahl);
       if (array[n] == element) return (n);
       rest = (rest >= 0) ? rest : -rest;
       rest = (array[n] <= element) ? rest : -rest;
       rest = signum * rest / 2;
   }
   rest = (element >= array[n]) ? signum : -signum;
   if (n + rest >= 0 && n + rest <= anzahl)
       n += ((element >= 0.5*(array[n] + array[n+rest]))?
				    rest + signum : rest - signum) / 2;
   return (n);
}


/* char *equalexp(double x, double dx, int digits, char *between)

     generates and returns an ascii string which
     containes the data of x and dx with  digits precision
     (digits = 1,2,3)) for darea in exponential format.
     the corresponding digits are output for area.
     between is a string wich will be inserted on output
     between x and dx 
     on return: NULL if an error ocured (wrong digit number etc)

   Friedrich Riess, Nov. 2002
*/

#include <stdio.h>
#include <math.h>


char form[132];

char *equalexp(double xx, double dxx, int digits, char *between)
{
  int count;
  char vorz;
  double x, dx, dxmin, dxmax;

  if(digits > 6 || digits < 1)
  {
    fprintf(stderr, "--> calling equalexp with illegal digit number (%d)\n", digits);
    return NULL;
  }
  if(dxx == 0.)
  {
    sprintf(form, "%16.8e %s%13.2e ", xx, between, dxx);
  }
  else
  {
    x = xx;
    dx = fabs(dxx);
    dxmax = pow(10., (double) digits) - 0.5;
    dxmin = pow(10., (double) (digits-1)) - 0.5;
    count = 0;
    while(dx > dxmax)
    {
      dx /= 10.;
      x /= 10.;
      ++count;
    }
    while(dx < dxmin)
    {
      dx *= 10.;
      x *= 10.;
      --count;
    }
    vorz = '+';
    if(count < 0)
    {
      vorz = '-';
      count = -count;
    }
    if(count == 0)
    {
      sprintf(form, "%11.0f.     %s%6.0f.     ", xx, between, dxx);
    }
    else
    {
      if(dxx < 0.) dx = -dx;
      sprintf(form, "%11.0f.e%c%2.2d %s%6.0f.e%c%2.2d ", x, vorz, count, between, dx, vorz, count);
    }
  }
  return form;
}

