/* $Id: readfft.c,v 2.4 2003/08/23 22:23:28 friedrich Exp $
   This routine reads spectra in a binary format:
   an ascii header is followed by binary data of doubleing numbers.

   header:
   first line:
     first 6 Bytes an integer indicating the begin of binary data
     followed by a space and the string "frequenzspektruum\n"
   2nd line:
     date and time
   3rd, 4th and 5th line: 
     fft - parameters
   6th line:
     6 Bytes spectrum length followed by " ka. ",
     first (6 Byte) " - " last (6 Byte) " Hz\n"  frequency
   7th line:
     multiplicator " Hanning         \n"
   further line:
     comments

  int readfft(
     char *filename:  Filename of Spectrum. If filename ends with .Z, it
                      will be assumed, that the file is compressed. 
                      An uncompressed file is generated under /usr/tmp
                      and used for reading. This file will be deleted
                      with a subsequent calls with an other filename. 
    int subspectrum:  ignored
    double *spectrum:  Array of the spectrum. Must have length of at least
                      ende - begin + 1
    int begin:        begin of the Spectrum
    int ende:         end of the spectrum
    int header:       if not zero: print text lines to the spectrum
    )
  return value:
    0 if the file is not found or if there was an error during reading
   <0 negativ value of last subspectrum if subspectrum does not exist
   >0 ende or last channel in the spectrum if ende is greater than 
      the last channel
  
   F. Riess, July 1998
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>

#define ERRMESS "\07--> readfft:"
#define BUFSIZE 1024

/* note: swapbytes is in file readosp.c */
void swapbytes(unsigned char *, int, int);

int readfft(char *filename, int subspectrum, double *spectrum, int begin, int ende, int header)
{
  int n, i, j, repeat, spec_begin, fd;
  int buffersize;
  char buff[BUFSIZE];
  static char filetmp[L_tmpnam > 130 ? L_tmpnam : 130] = "";
  static char fileold[130] = " ", str[131];
  static int compressfile = 0, spectrumsize;
  static double *data = NULL;
    
/* the following statement is used to avoid failure with -Werror */
  i = subspectrum;
/* check if file has already been accessed */
  if(strcmp(fileold, filename))
  {
    if(compressfile) (void) unlink(filetmp);
    if(data) free(data);
    data = NULL;
    strcpy(fileold, filename);
    if(access(filename, 00) == -1)
    {
      fprintf(stderr, "%s file %s can not be accessed\n", ERRMESS, filename);
      return 0;
    }
    j = strlen(filename) - 1;
    if(!strcmp(&filename[j-1], ".Z")) j = 0;
    else if(!strcmp(&filename[j-2], ".gz")) j = -1;
    if(j > 0)
      strcpy(filetmp, filename);
    else
    { 
      strcpy(filetmp, "/tmp/gspfftXXXXXX");
      j = mkstemp(filetmp);
      if(j < 0) return 0;
      close(j);
      sprintf(str, "zcat %s > %s\n", filename, filetmp);
      system(str);
    }
    if((fd = open(filetmp, O_RDONLY)) == -1)
    {
      fprintf(stderr, "%s can not open  %s\n", ERRMESS, filetmp);
      return 0;
    }
/* read header */
    for(repeat = 0;;++repeat)
    {
      n = read(fd, buff, BUFSIZE);
      /* extract spectrum begin */
      if(!repeat)
      {
        strncpy(str, buff, 6);
        str[6] = '\0';
        sscanf(str, "%d", &spec_begin);
        if(header) printf("%d\n", spec_begin);
      }
      /* print header if wanted, extract spectrum length */
      for(i = repeat ? 0 : 7, n = 0; i < (spec_begin > (repeat+1)*BUFSIZE ? BUFSIZE : spec_begin); ++i) 
      {
        if(header)  printf("%c", buff[i]);
        if(buff[i] =='\n') ++n;
        if(n == 5)
        {
           strncpy(str, &buff[i+1], 6);
           ++n;
        }
      }
      if(!repeat) 
      {
        str[6] = '\0';
        sscanf(str, "%d", &spectrumsize);
       if(header) printf("length of spectrum: %d\n", spectrumsize);
      }
      if(spec_begin <= (repeat+1)*BUFSIZE) break;
    }
    buffersize = spectrumsize * sizeof(double);
    if((data = (double*) malloc(buffersize)) == NULL)
    {
      fprintf(stderr, "%s can not allocate buffer for spectrum\n", ERRMESS);
      return 0;
    }
    lseek(fd, spec_begin, SEEK_SET);
    n = read(fd, data, buffersize);
    close(fd);
    if(n != buffersize)
    {
      fprintf(stderr, "%s error in reading spectrum (%d Bytes, should %d)\n",
          ERRMESS, n, buffersize);
    }
    /* check for little or big endian */
    for(i = n = 0; i < spectrumsize && !n ; ++i)  n = data[i] > 1.e+10;
    if(n)  swapbytes((unsigned char *) data, buffersize, 4); 
  }
/* Spectrum is in buffer, transfer it to extern buffer */
  if(ende >= spectrumsize) ende = spectrumsize;
  n = begin;
  for(i = 0; i <= (ende - begin); ++i, ++n)
    spectrum[i] = data[n];
  return ende;
}  


/* Fortran interface */
int readfft_(char *fi, int *su, double *sp, int *be, int *en, int *header, int fl)
{
  char file[130];
  int i;
  int readfft(char *, int, double *, int, int, int);

  for(i = 0; i < fl; ++i)
  {
    file[i] = fi[i];
    if(fi[i] == ' ') break;
  }
  file[i] = '\0';
  i = readfft(file, *su, sp, *be, *en, *header);
  return i;
}
