/*  package for TEKTRONIX GRAFICS
   $Id: Tekgraflib0.c,v 1.2 2000/05/19 12:05:34 riess Exp $
   Low level routinen
      int openpl(), int openpf(filename), void closepl(),
      void move(x, y), void cont(x, y), void label(x, y)
      void label(string),
      void set_color(farbe), void get_color()
    haben als Argument einfache integer (int) und setzen einen Wertebereich
    von 0 <= x < 8192 und 0 <= y < 6144 voraus
*/

#include "grafik.h"
#include <stdio.h>
#include <string.h>
#include <time.h>

#define ETX '\003'
#define LF '\014'
#define CAN '\030'
#define SUB '\032'
#define ESC '\033'
#define EM '\031'
#define FS '\034'
#define GS '\035'
#define RS '\036'
#define US '\037'

#define OPEN_DIS "\033[?38h"      /* switch to Tektronix mode */
#define FILESAVE "gasplot.bak"
#define PLOTDEFAULT "kl"

#define K_FAKTOR 0.0019    /* Scaling Faktor fuer Portrait */
#define L_FAKTOR 0.0027    /* Scaling Faktor fuer landscape */
#define P_FAKTOR 0.0551    /* Scaling Faktor fuer Postscript */
#define P_OFFSET 70        /* Offset fuer Postscript */
#define STRICH 2.54/300.   /* Minimum Strichstaerke */

FILE *dis = 0;
FILE *plot = 0;

/* Ausgaben: Hp = Hewlett Packard Plotter
	     kp = Kyocera portrait
	     kl = Kyocera landscape
             ps = Postscript
  default ist ps; die Ausgabe Form kann ueber den Plotter Filename
  kontrolliert werden: filenamen beginnend mit h produzieren hp Format,
  solche beginnend mit p kp - Format, solche mit l kl;
*/
enum {hp, kp, kl, ps} plotform;

double xslope = 0, yslope, xoffset, yoffset;
static double strichart;
static int bold_mode = 0;
char offset;
int x_old, y_old;
int ps_count;

void draw(int, int);
int openpl(void)
{
  void gr_open_file(void);

  if(xslope == 0.)
  {
    xslope = 1. / 4096.; xoffset = 0.;
    yslope = 1. / 3072.; yoffset = 0.;
  }
  x_old = y_old = -1;
  if(!dis)
  { if((dis = fopen("/dev/tty", "w")) == (FILE *) 0) return 1;
    fputs(OPEN_DIS, dis);  fputc(GS, dis);
  }
  if(plot)
  {
    switch (plotform)
    { case hp:  fputs("\033.R\033.N;19:\033.I128;;17:IN;SP3;PU;\n", plot);
		fputs("SC 0 8192 0 6144;\n", plot);
		break;
      case kp:
      case kl:
                fputs("!R! RES; SPSZ 8; UNIT C;\n", plot);
		fputs(" STM 3.0; SLM 3.0;\n", plot);
		if(plotform == kp)  fputs("SPO P; FONT 1;\n", plot);
		else fputs("STM 2.5; SLM 3.7; SPO L; FONT 17;\n", plot);
      default: gr_open_file();
    }
  }
  linemod("normal"); linemod("solid");
  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  void gr_open_file(FILE *)
  Schreibt den header fuer die Postscript datei
--------------------------------------------------------------------*/
void gr_open_file(void)
{
  char str[120];
  time_t zeit;
  fputs("%!PS-Adobe-1.0\n", plot);
  fputs("%%Creator: Tekgrafik\n", plot);
  (void) time(&zeit);
  sprintf(str, "%%%%CreationDate: %s", ctime(&zeit));
  fputs(str, plot);
  sprintf(str, "%%%%BoundingBox: %d %d %d %d\n", P_OFFSET, P_OFFSET, 
	  (int) (P_OFFSET + P_FAKTOR*8192), (int) (P_OFFSET + P_FAKTOR*6144));
  fputs(str, plot);
  fputs("%%EndComments\n", plot);
/* Definiere eine Elipse */
  fputs("/ellipsedict 8 dict def\nellipsedict /mtrx matrix put\n", plot);
  fputs("/ellipse\n{ ellipsedict begin\n  /endangle exch def\n", plot);
  fputs("  /startangle exch def\n  /yrad exch def  /xrad exch def\n", plot);
  fputs("  /y exch def\n  /x exch def\n", plot);
  fputs("  /savematrix mtrx currentmatrix def\n  x y translate\n", plot);
  fputs("  xrad yrad scale\n  0 0 1 startangle endangle arc\n", plot);
  fputs("  savematrix setmatrix\n end\n} def\n", plot);
  fputs("/m {moveto} bind def\n", plot);
  fputs("/l {lineto} bind def\n", plot);
  fputs("/r {rlineto} bind def\n", plot);
  ps_count = 0;
}


int openpf(char* filename)
{
  if(strcmp(filename, ""))
  { plot = fopen(filename, "w");
    if(*filename == 'h') plotform = hp;
    else if(*filename == 'p') plotform = kp;
    else if(*filename == 'l') plotform = kl;
    else plotform = ps;
  }
  else
  { plot = fopen(FILESAVE, "w");
    plotform = ps;
  }
  return openpl();
}

void closepl(int mode)
{ if(dis)
  { label("", 1);
    fflush(dis);
    fputc(ESC, dis); fputc(GS, dis);
    fputc(CAN, dis);
    fputc(ESC, dis); fputc(ETX, dis);
    fflush(dis);
    fflush(dis);
    fclose(dis);
    dis = (FILE *) 0;
    if(mode)
    {
      fprintf(stdout, "> ");
      fflush(stdout);
      (void) getchar();
    }
  }
  if(plot)
  { switch(plotform)
    { case hp:  fputs("PU; SP;\n", plot);
		break;
      case kp:
      case kl:  fputs("RES; EXIT;\n", plot);
                break;
      default:  fputs("stroke showpage\n", plot);
    }
    fclose(plot);
    plot = (FILE *) 0;
  }
}

void erase(void)
{
  x_old = y_old = -1;
  if (dis)
  { fputc(ESC, dis); fputc(LF, dis);
  }
  fflush(dis);
}

int get_color(int ground)
{
  return 1;
}

void set_color(int farbe)
{
}

void point(int x, int y)
{
  if(!(x == x_old && y == y_old))
  { x_old = x; y_old = y;
    if(dis) fputc(FS, dis);
    if(plot)
      switch(plotform)
      { case hp:  fputs("PU ", plot);
	          break;
        case kp:
        case kl:  fputs("MAP ", plot);
	          break;
        default:  fputs("stroke\n", plot);
      }
    draw(x, y);
    if(plot)
      switch(plotform)
      { case hp:  fputs(" PD ;\n", plot);
	          break;
        case kp:
        case kl: fputs("; SPD 0.03; CIR 0.02; SPD ", plot);
	         fprintf(plot, "%.3f;\n", strichart);
	         break;
        default:  fputs(" m\n 1 0 r\n", plot);
      }
  }
}

void move(int x, int y)
{
  if(dis) fputc(GS, dis);
  if(plot)
    switch(plotform)
    { case hp:  fputs("PU ", plot);
 	  break;
      case kp:
      case kl:  fputs("MAP ", plot);
         break;
      default:  fputs("stroke\n", plot);
    }
  draw(x, y);
  x_old = x; y_old = y;
  if(plot)
  { if(plotform == ps) fputs(" m\n", plot);
    else fputs(";\n", plot);
  }
}

void cont(int x, int y)
{
  if(!(x == x_old && y == y_old))
  {
    if(plot)
    {
      switch(plotform)
      { case hp:  fputs("PD ", plot);
	  	  break;
        case kp:
        case kl:  fputs("DAP ", plot);
                  break;
        default:  if(++ps_count > 1024)
	          { ps_count = 0;
                    fputs("stroke\n", plot);
                    draw(x_old, y_old);
		    fputs(" m\n", plot);
	 	  }
      }
    }
    draw(x, y);
    x_old = x; y_old = y;
    if(plot)
    { if(plotform == ps) fputs(" l\n", plot);
      else fputs(";\n", plot);
    }
  }
}

void draw(int x, int y)
{ 
  if(dis)
  { fputc(0x20 | ((y/256) & 0x1f), dis);
    fputc(0x60 | ( y/8 & 0x1f)   , dis);
    fputc(0x20 | ((x/256) & 0x1f), dis);
    fputc(0x40 | ( x/8 & 0x1f)   , dis);
  }
  if(plot)
    switch(plotform)
    { case hp: fprintf(plot, "%d %d", x, y); break;
      case kp: fprintf(plot, "%.3f, %.3f", K_FAKTOR * x, K_FAKTOR * (6144 - y));
	       break;
      case kl: fprintf(plot, "%.3f, %.3f", L_FAKTOR * x, L_FAKTOR * (6144 - y));
               break;
      default: fprintf(plot, "%.2f %.2f", P_FAKTOR * x + P_OFFSET,
                 P_FAKTOR * y + P_OFFSET);
    }
}

void label(char *string, int size)
{
  int i, font;
  double width, height;
  i = (size >= 0)? size : 0; i = (i <= 3)? i : 3;
  if(dis)
  { fputc(ESC, dis); fputc(';' - i, dis);
    fputc(US , dis); fputs(string, dis);
  }
  if(plot && *string)
  { font = 0;
    switch(plotform)
    { case hp:  fputs("PU ", plot);
		fputs("SI", plot);
		switch (i)
		{ case 1:   width = 0.140; height = 0.250; break;
		  case 2:   width = 0.207; height = 0.360; break;
		  case 3:   width = 0.235; height = 0.440; break;
		  default:  width = 0.127; height = 0.200;
		}
		fprintf(plot, " %.3f %.3f;\n", width, height);
		fputs("LB", plot); fputs(string, plot); fputs("\003;\n", plot);
		break;
      case kp:  switch(i)
		{ case 1:  if(offset) font = 48;
			   else       font =  7;
			   break;
		  case 2:  if(offset) font = 62;
			   else       font = 15;
			   break;
		  case 3:  if(offset) font =  9;
			   else       font =  8;
			   break;
		  default: if(offset) font = 65;
			   else       font = 16;
		}
      case kl: if(font != i)
		  switch(i)
		{ case 1:  if(offset) font = 71;
			   else       font = 23;
			   break;
		  case 2:  if(offset) font = 67;
			   else       font = 17;
			   break;
		  case 3:  if(offset) font = 29;
			   else       font = 17;
			   break;
		  default: if(offset) font = 73;
			   else       font = 25;
		}
		fprintf(plot, "FONT %d;\n", font);
		fprintf(plot, "TEXT \'%s\';\n", string);
                break;
      default:  font = 6 + 2 * (1 + (i % 4));
                if(bold_mode) fprintf(plot, "/Times-Bold");
                else          fprintf(plot, "/Times-Roman");
                fprintf(plot, " findfont\n%d scalefont\nsetfont\n", font);
                fprintf(plot, "(%s) show\n", string);
    }
  }
}


void linemod(char *mode)
{
  static char *modes[] = {"dotted", "shortdashed", "longdashed",
			  "dotdashed", "solid", "normal", "bold", ""};
  static char *dash_list[] = {"1 3 1 3",
                              "4 4 4 4",
                              "8 8 8 8",
                              "1 3 5 3",
                              "",
	                     };
  static char cmode;
  int m;
  for(m = 0; *modes[m] != '\0'; ++m)
    if(!strncmp(mode, modes[m], 4)) break;
  m = m > 6 ? 4 : m;
  if(m == 5) bold_mode = 0;
  else if(m == 6) bold_mode = 1;     
  if (dis)
  {
    switch (m)
    {
      case 0:   cmode = 'a'; break;
      case 1:   cmode = 'c'; break;
      case 2:   cmode = 'd'; break;
      case 3:   cmode = 'b'; break;
      case 4:   cmode = '`'; break;
      case 5:   offset = '\0'; break;
      case 6:   offset = 'i' - 'a'; break;
      default:  cmode = '`';
    }
    fputc(ESC, dis); fputc(cmode+offset, dis);
  }
  if(plot)
  {  switch(plotform)
    { case hp: fputs("LT ", plot);
	       if(m >= 0 && m <= 3)  fprintf(plot, "%d;\n", m + 1);
	       else fputs(";\n", plot);
	       if(offset) fputs("PT 0.6;\n", plot);
	       else       fputs("PT 0.3;\n", plot);
	       break;
      case kp:
      case kl: if(m < 5)
               { strichart = STRICH * (m + 1);
	         if(offset) strichart = STRICH * ( m + 5);
	         fprintf(plot, "SPD %.3f;\n", strichart);
	       }
               break;
      default: if(m < 5) fprintf(plot, "stroke\n [%s] 0 setdash\n",
                                           dash_list[m]);
               else fprintf(plot, "stroke\n%.1f setlinewidth\n",
                                           m == 5 ? 0.5 : 1.0);
    }
  }
}

void flushpl(void)
{
  fflush(dis);
}

void set_window_size(double a, double b)
{
}
