/* Grafik Paket fuer X-Window Grafik mit Postscript Ausagbe
  $Id: Xgrafik.c,v 1.15 2005/06/01 10:05:42 friedrich Exp friedrich $
  F. Riess,  Februar 1993
*/

#define PS_COLOR

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "Xgrafik.h"
#include "grafik.h"

#define MAXCHAR 10
#define FP_XMIN 30.   /* Maximale Koordinaten fuer Postscript */
#define FP_YMIN 50.
#define FP_WIDTH 530.
#define FP_HEIGHT 700.

Display *x_gr_dpy = NULL;
Window x_gr_wd, x_gr_root, w_msg;
Pixmap x_gr_pixmap;
Screen *x_gr_screen;
int x_gr_int_screen;
GC x_gr_gc;
XWindowAttributes x_gr_xwa;
XSizeHints *sizehints;

double x_gr_offset_x, x_gr_offset_y, x_gr_slope_x, x_gr_slope_y;
double x_gr_rel_width = 0., x_gr_rel_height = 0.;
static int x_gr_coord_x(double), x_gr_coord_y(double);

FILE *x_gr_fp = (FILE *) 0;
char *x_gr_filename;
char x_gr_title_str[] = {"Grafikfenster"};
char *x_gr_title = x_gr_title_str;
double x_gr_x_min, x_gr_x_max, x_gr_y_min, x_gr_y_max;
double x_fp_offset_x, x_fp_offset_y, x_fp_slope_x, x_fp_slope_y;
double x_fp_width = 0., x_fp_height = 0.;
static double x_fp_coord_x(double), x_fp_coord_y(double);

static double winkel_zu_hori(double, double, double, double);
static void closepl_x(void);
static int x_get_Event(int Eventtype, int wait);

int x_gr_current_x, x_gr_current_y;
int x_gr_width, x_gr_height, x_gr_depth;
int x_gr_color_f = -1, x_gr_color_b = -1;
int x_gr_count = 0;

char *x_gr_line_styles[]={"solid",
	                  "dotted",
                          "shortdashed", 
                          "longdashed",
                          "dotdashed",
                         };

/* Farb Definitionen, siehe auch grafik.h */

struct  {char *name; unsigned long wert;}    /* RGB Werte   */
   x_gr_farbe[] = {{"Black",  0},             /*   0   0   0 */
                   {"White", 0},              /* 255 255 255 */
                   {"Red", 0},                /* 255   0   0 */
                   {"Green", 0},              /*   0 255   0 */
                   {"Blue", 0},               /*   0   0 255 */
                   {"palevioletred", 0},      /* 219 112 147 */
                   {"mediumvioletred", 0},    /* 199  21 133 */
                   {"palegreen", 0},          /* 152 251 152 */
                   {"seagreen", 0},           /*  49 139  87 */
                   {"lightskyblue", 0},       /* 135 206 250 */
                   {"mediumblue", 0},         /*   0   0 205 */
                   {"yellow", 0},             /* 255 255   0 */
                   {"violet", 0},             /* 238 130 238 */
                   {"orange", 0},             /* 255 165   0 */
                   {"darkgoldenrod", 0},      /* 184 134  11 */
                   {"DimGray", 0},            /* 105 105 105 */
                  };


#define x_gr_MAX_COLORS (sizeof(x_gr_farbe)/sizeof(x_gr_farbe[0]))

/* entsprechende Farbtabelle fuer Postscriptausgabe */
#ifdef PS_COLOR
 struct {unsigned char r, g, b;}
   x_gr_fp_color[] = {{   0,   0,   0},
                      { 255, 255, 255},
                      { 255,   0,   0},
                      {   0, 255,   0},
                      {   0,   0, 255},
                      { 219, 112, 147},
                      { 199,  21, 133},
                      { 152, 251, 152},
                      {  49, 139,  87},
                      { 135, 206, 250},
                      {   0,   0, 205},
                      { 255, 255,   0},
                      { 238, 130, 238},
                      { 255, 165,   0},
                      { 184, 134,  11},
                      { 105, 105, 105},
                  };
#else
/* oder entsprechende Graustufen fuer Schwarz-Weiss Laser Drucker */
double x_gr_fp_color[] = {0.0, 1.0, 0.5, 0.7, 0.6, 0.6, 0.4, 0.8,
                          0.4, 0.7, 0.3, 0.9, 0.3, 0.4, 0.35,  0.2}; 
#endif

/* Font Definitionen */
int x_gr_fontsize, x_gr_old_fontsize;
#ifdef HP
char *x_gr_font_names[] = {
                   "-*-*-*-r-*-*-*-100-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-140-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-180-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-240-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-80-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-100-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-140-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-200-*-*-*-*-*-*"
		          };
#else
char *x_gr_font_names[] = {
                   "-*-new century schoolbook-medium-r-*-*-*-140-*-*-*-*-*-*",
                   "-*-new century schoolbook-medium-r-*-*-*-180-*-*-*-*-*-*",
                   "-*-new century schoolbook-medium-r-*-*-*-230-*-*-*-*-*-*",
                   "-*-new century schoolbook-medium-r-*-*-*-300-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-80-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-100-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-140-*-*-*-*-*-*",
                   "-*-*-*-r-*-*-*-300-*-*-*-*-*-*"
	          };
#endif

XFontStruct *x_gr_font_info;
int x_fp_old_fontsize;
int x_fp_fontsize[] = {6, 8, 10, 14};


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   winkel_zu_hori(x1, yy1, x2, yy2)
   berechnet den Winkel zur Horizontalene zwischen den Punkten
     (x1, yy1) und (x2, yy2)

   wird von arc1 und arc2 benoetigt
--------------------------------------------------------------------*/
static double winkel_zu_hori(double x1, double yy1, double x2, double yy2)
{
  double w;
  if(x1 == x2 && yy1 == yy2) w = 0.;
  else
    w = (w = atan2(yy2 - yy1, x2 - x1)) > 0 ? w : (w + 2 * M_PI); 
  return(w);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   x_gr_coord_x(x)
   berechnet die Transformation ins grafische Koordinatensystem
--------------------------------------------------------------------*/
static int x_gr_coord_x(double x)
{
  return (int)(x_gr_slope_x * x + x_gr_offset_x);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   x_gr_coord_y(x)
   berechnet die Transformation ins grafische Koordinatensystem
--------------------------------------------------------------------*/

static int x_gr_coord_y(double y)
{
  return (int)(x_gr_slope_y * y + x_gr_offset_y);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   x_fp_coord_x(x)
   berechnet die Transformation ins postscript Koordinatensystem
--------------------------------------------------------------------*/
static double x_fp_coord_x(double x)
{
  return (x_fp_slope_x * x + x_fp_offset_x);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   x_fp_coord_y(x)
   berechnet die Transformation ins postscript Koordinatensystem
--------------------------------------------------------------------*/

static double x_fp_coord_y(double y)
{
  return (x_fp_slope_y * y + x_fp_offset_y);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   int check_mouse(void)
   the routine checks for mouse clicks

   it return  0 if no mouse click has been detected
              1 otherwise
                no further action will be taken!
--------------------------------------------------------------------*/
int check_mouse(double *x, double *y)
{
  extern Display *x_gr_dpy;
  extern double x_gr_offset_x, x_gr_offset_y, x_gr_slope_x, x_gr_slope_y;
  extern int x_gr_current_x, x_gr_current_y;
  Window root, child;
  int irx, iry;
  unsigned int key_button;
  int flag = 0;
  
  if(x_gr_dpy)
  {
    if(x_get_Event(ButtonPress, 0))
      {
        XQueryPointer(x_gr_dpy, x_gr_wd, &root, &child, &irx, &iry,
            &x_gr_current_x, &x_gr_current_y, &key_button);
        *x = (x_gr_current_x - x_gr_offset_x) / x_gr_slope_x;
        *y = (x_gr_current_y - x_gr_offset_y) / x_gr_slope_y;
       flag = 1;
      }
  }
  return flag;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  int check_remap(void)
  routine checks if the window was iconized and returns the value 1
    if it has been remapped, to allow for a redrawing
  it returns 0 if no change in mapping was found
--------------------------------------------------------------------*/
int check_remap(void)
{
  extern Display *x_gr_dpy;
  int flag = 0;

  if(x_gr_dpy)
  {
    flag = x_get_Event(MapNotify, 0);
  }
  return flag;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   int check_resize(void)
   the routine checks for a change in the size of the window

   it returna 0 if no change has been detected
              1 otherwise
                no further action will be taken!
--------------------------------------------------------------------*/
int check_resize(void)
{
  extern Display *x_gr_dpy;
  int flag = 0;
  
  if(x_gr_dpy)
  {
    flag = x_get_Event(ResizeRequest, 0);
  }
  return flag;
}
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   int check_expose(void)
   the routine checks for a change in the size of the window

   it returna 0 if no change has been detected
              1 otherwise
                no further action will be taken!
--------------------------------------------------------------------*/
int check_expose(void)
{
  extern Display *x_gr_dpy;
  int flag = 0;
  
  if(x_gr_dpy)
  {
    flag = x_get_Event(Expose, -1);
  }
  return flag;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   changes the focus to either the root Window (target = "root"
   or the graphics window (anything else)
--------------------------------------------------------------------*/
void set_focus(char *target)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd ; /*, x_gr_root; */
  Time zeit = CurrentTime;
  Window focus;
  int revert_to;

  if(x_gr_dpy)
  {
    if(*target == 'r') XSetInputFocus(x_gr_dpy, PointerRoot, RevertToPointerRoot, zeit);
    else  XSetInputFocus(x_gr_dpy, x_gr_wd, RevertToParent, zeit);
    XGetInputFocus(x_gr_dpy, &focus, &revert_to);
  }
}

  

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   X- Window Grafik:   int openpl()
     eroeffnet die Grafik: 
        eroeffnet ein Fenster, eine Grafik Drawable und ein Pixmap
--------------------------------------------------------------------*/
int openpl(void)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd, x_gr_root;
  extern Pixmap x_gr_pixmap;
  extern Screen *x_gr_screen;
  extern int x_gr_int_screen;
  extern GC x_gr_gc;
  extern int x_gr_width, x_gr_height, x_gr_depth;
  extern int x_gr_color_b, x_gr_color_f;
  extern XSizeHints *sizehints;

  void x_gr_open_file(void);

  unsigned long valuemask;
  XWMHints   *wmhints;
  XClassHint *classhints;
  XSetWindowAttributes attributes;
  XTextProperty windowname, iconname;
  int error;

  if(!x_gr_dpy)
  {
    x_gr_dpy = XOpenDisplay(NULL);
    if(!x_gr_dpy)
    {
      fprintf(stderr,"--> sorry, no connection to X-Server!\n");
      return -1;
    }
    x_gr_count = 0;
    x_gr_int_screen = DefaultScreen(x_gr_dpy);
    x_gr_root = XRootWindow(x_gr_dpy, x_gr_int_screen);
    (void) get_color(-1);
    x_gr_screen = XScreenOfDisplay(x_gr_dpy, x_gr_int_screen);
/* get size of window relativ to the screen size */
    x_gr_width = WidthOfScreen(x_gr_screen);
    x_gr_height = XHeightOfScreen(x_gr_screen);
    if(x_gr_rel_width <= 0.)
    { if(x_gr_rel_height <= 0.) x_gr_rel_width = 0.8;
      else x_gr_rel_width = x_gr_rel_height * 0.6 * x_gr_width / x_gr_height;
    }
    if(x_gr_rel_width >= 1.) x_gr_rel_width = 1.;        
    if(x_gr_rel_height <= 0.)
       x_gr_rel_height = 0.75 * x_gr_rel_width * x_gr_width / x_gr_height;
    if(x_gr_rel_height >= 1.) x_gr_rel_height = 1.;
    x_gr_width *= x_gr_rel_width; 
    x_gr_height *= x_gr_rel_height;

/* get foreground and background color */
    if(x_gr_color_b < 0) x_gr_color_b = 1;
    if(x_gr_color_f < 0)
      { x_gr_color_f = 0;
      if(x_gr_color_b == x_gr_color_f) x_gr_color_f = !x_gr_color_b;
    }
    x_gr_depth = XDefaultDepth(x_gr_dpy, x_gr_int_screen);
/* allocate memory */
    error = 0;
    if(!(sizehints=XAllocSizeHints()))  ++error;
    if(!(wmhints=XAllocWMHints()))      ++error;
    if(!(classhints=XAllocClassHint())) ++error;
    if(error)
    {
      fprintf(stderr,"Xgrafik: failure allocating memory\n");
      x_gr_dpy = NULL; return (-1);
    }
    error = 0;
    if(XStringListToTextProperty(&x_gr_title, 1, &windowname) == 0) ++error;
    if(XStringListToTextProperty(&x_gr_title, 1, &iconname) == 0)   ++error;
    if(error)
    {
      fprintf(stderr,"Xgraphics: failure allocating structure\n");
      x_gr_dpy = NULL; return (-1);
    }

/* create window */  
    x_gr_wd =  XCreateSimpleWindow(x_gr_dpy, x_gr_root,
                0, 0, x_gr_width, x_gr_height,                      /* coords */
                20, x_gr_farbe[x_gr_color_f].wert, x_gr_farbe[x_gr_color_b].wert);
    x_gr_pixmap = XCreatePixmap(x_gr_dpy, x_gr_wd, x_gr_width, x_gr_height, x_gr_depth); 

/* set properties */
    wmhints->initial_state = NormalState;
    wmhints->input = True;
    wmhints->flags = StateHint | InputHint;
/* set window size, do not allow resizing */
    sizehints->flags = PPosition | PSize | PMinSize | PMaxSize;
    sizehints->max_width =  x_gr_width;
    sizehints->max_height = x_gr_height;
    sizehints->min_width =  x_gr_width;
    sizehints->min_height = x_gr_height;

    XSetWMProperties(x_gr_dpy, x_gr_wd, &windowname, &iconname, NULL, 0, sizehints,
		   wmhints, classhints);

    valuemask = CWBackingStore | CWSaveUnder | CWBitGravity;
    attributes.bit_gravity = NorthWestGravity;
    attributes.backing_store    = WhenMapped;
    attributes.save_under = True;
    XChangeWindowAttributes(x_gr_dpy, x_gr_wd, valuemask, &attributes);

/* Select input options */
    XSelectInput(x_gr_dpy, x_gr_wd,
	    OwnerGrabButtonMask |   
            ButtonPressMask | ButtonReleaseMask |         /* Mouse */
            KeyPressMask    |                          /* Keyboard */
            ExposureMask);                    /* Window - Exposure */


    x_gr_gc = XCreateGC(x_gr_dpy, x_gr_wd,0, NULL);

/* clear pixmap */
    XSetForeground(x_gr_dpy, x_gr_gc, x_gr_farbe[x_gr_color_b].wert);
    XFillRectangle(x_gr_dpy, x_gr_pixmap, x_gr_gc, 0, 0, x_gr_width, x_gr_height);
    XSetForeground(x_gr_dpy, x_gr_gc, x_gr_farbe[x_gr_color_f].wert);
    XSetBackground(x_gr_dpy, x_gr_gc, x_gr_farbe[x_gr_color_b].wert);

    XMapWindow(x_gr_dpy, x_gr_wd);

/* wait on Expose Event */
    while(!x_get_Event(Expose, 1));
  }
  if(x_gr_filename) x_gr_open_file();
  set_f_scale(0., 0., 1024., 768.);
  linemod("solid");
  linemod("normal");
  label("", 1);
  return(0);
} 

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   int openplt( char *title)
     opens the grafics with an own window name "title" 
--------------------------------------------------------------------*/
int openplt(char *title)
{
  x_gr_title = title;
  return openpl();
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  openpf(filename)
  Eroeffnet die Grafik und generiert zusaetzlich eine Datei der
  Grafikprimitiven unter dem Namen "filename"
    fall kein Filename angegeben wurde, wird "xwindowplt.bak" benutzt
--------------------------------------------------------------------*/

int openpf(char *s)
{
  extern char *x_gr_filename;
  static char file[100];
  if(strlen(s)) (void) strcpy(file, s);
  else          (void) strcpy(file, "xwindowplt.ps");
  x_gr_filename = file;
  return openpl();
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  openpft(filename, title)
  Eroeffnet die Grafik und generiert zusaetzlich eine Datei der
  Grafikprimitiven unter dem Namen "filename"
    fall kein Filename angegeben wurde, wird "xwindowplt.bak" benutzt
--------------------------------------------------------------------*/

int openpft(char *s, char *title)
{
  x_gr_title = title;
  return openpf(s);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  void x_gr_open_file(void)
  Eroeffnet die Ausgabe Datei fuer das Retten der Grafikbefehle
  Grafikprimitiven unter dem Namen "filename"
    fall kein Filename angegeben wurde, wird "xwindowplt.bak" benutzt
--------------------------------------------------------------------*/
void x_gr_open_file(void)
{
  extern FILE *x_gr_fp;
  extern char *x_gr_filename;
  extern int x_gr_width, x_gr_height;
  extern double x_fp_width, x_fp_height;
  time_t zeit;
  char tmpfile[120];

  sprintf(tmpfile, "%s.", x_gr_filename);  
  if((x_gr_fp = fopen(tmpfile, "w")))
  {
    x_fp_width = FP_WIDTH*x_gr_rel_width;
    x_fp_height =  x_gr_height * x_fp_width / x_gr_width;
    if(x_fp_height > FP_HEIGHT)
    { x_fp_height = FP_HEIGHT * x_gr_rel_height;
      x_fp_width = x_fp_height * x_gr_width / x_gr_height;
    }
    fprintf(x_gr_fp, "%%!PS-Adobe-1.0\n");
    fprintf(x_gr_fp, "%%%%Creator: Xgrafik\n");
    (void) time(&zeit);
    fprintf(x_gr_fp, "%%%%CreationDate: %s", ctime(&zeit));
    fprintf(x_gr_fp, "%%%%BoundingBox: %d %d %d %d\n", (int)floor(FP_XMIN), 
       (int)floor(FP_YMIN), (int)ceil(FP_XMIN+x_fp_width), (int)ceil(FP_YMIN+x_fp_height));
    fprintf(x_gr_fp, "%%%%EndComments\n");
    fprintf(x_gr_fp, "%%%%EndSetup\n");
    fprintf(x_gr_fp, "%%%%Page: 1 1\n");
/* Definiere eine Elipse */
    fprintf(x_gr_fp, "/ellipsedict 8 dict def\nellipsedict /mtrx matrix put\n");
    fprintf(x_gr_fp, "/ellipse\n{ ellipsedict begin\n  /endangle exch def\n");
    fprintf(x_gr_fp, "  /startangle exch def\n  /yrad exch def  /xrad exch def\n");
    fprintf(x_gr_fp, "  /y exch def\n  /x exch def\n");
    fprintf(x_gr_fp, "  /savematrix mtrx currentmatrix def\n  x y translate\n");
    fprintf(x_gr_fp, "  xrad yrad scale\n  0 0 1 startangle endangle arc\n");
    fprintf(x_gr_fp, "  savematrix setmatrix\n end\n} def\n");
  }
  x_fp_old_fontsize = -1;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  closepl(modus)
  schliesst die Grafik mit modus = 0 sofort, sonst mit einem
  Abfragefenster
--------------------------------------------------------------------*/
void closepl(int modus)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern GC x_gr_gc;

  int width, height;

  if(x_gr_dpy)
  {
    if(modus)
    {
      extern XGCValues xgcv;
      double x, y;
      static char sclose[] = {"close"};
  
      xgcv.line_style = LineSolid;
      xgcv.line_width = 3;
      XSetLineAttributes(x_gr_dpy, x_gr_gc, xgcv.line_width, xgcv.line_style, 
                     xgcv.cap_style, xgcv.join_style);
      width = x_gr_width > 100. ? 100 : x_gr_width;
      height = x_gr_height > 15 ? 15 : x_gr_height;
      set_color(x_gr_color_f);
      XFillRectangle(x_gr_dpy, x_gr_wd, x_gr_gc, 0, 0, width, height);
      XFillRectangle(x_gr_dpy, x_gr_pixmap, x_gr_gc, 0, 0, width, height);
      set_color(x_gr_color_b);
      label("", 2);
      XDrawString(x_gr_dpy, x_gr_wd, x_gr_gc, width/4, height - 3, sclose, strlen(sclose));
      XDrawString(x_gr_dpy, x_gr_pixmap, x_gr_gc, width/4, height - 3, sclose, strlen(sclose));
      flushpl();
      do
      {
         while(get_mouse_position(&x, &y));
      } while(x_gr_current_x > width || x_gr_current_y > height);
    }
  }
  closepl_x();
} 


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   closepl_x()
   schliesst das Fenster und setzt die Variablen zurueck
--------------------------------------------------------------------*/
static void closepl_x(void)
{
  extern Display *x_gr_dpy;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_color_f, x_gr_color_b;
  extern FILE *x_gr_fp;

  if(x_gr_dpy)
  {
    if(x_gr_font_info) XFreeFont(x_gr_dpy, x_gr_font_info);
    XFreeGC(x_gr_dpy, x_gr_gc);
    XFreePixmap(x_gr_dpy, x_gr_pixmap);
    XCloseDisplay(x_gr_dpy);
  }
  x_gr_dpy = 0;
  x_gr_color_b = x_gr_color_f = -1;
  x_gr_rel_width = x_gr_rel_height = 0.;
  x_gr_font_info = 0;
  if(x_gr_fp)
  { 
    closeps(0);
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   closeps
   closes the postscript file, renames it
   and opens it again if mode is none zero.
--------------------------------------------------------------------*/
void closeps(int mode)
{
  extern FILE *x_gr_fp;
  char tmpcmd[120];

  if(x_gr_fp)
  {
    fprintf(x_gr_fp, "stroke showpage\n");
    fprintf(x_gr_fp, "%%%%Trailer\n");
/* ghostview does not like this statement if ps -files sre to be
included as pictures
    fprintf(x_gr_fp, "end\n");
*/
    fprintf(x_gr_fp, "%%%%Pages 1\n");
    fprintf(x_gr_fp, "%%%%EOF\n");
    fclose(x_gr_fp);
    x_gr_fp = NULL;
    sprintf(tmpcmd, "mv %s. %s", x_gr_filename, x_gr_filename);
    system(tmpcmd); 
    if(mode) x_gr_open_file();
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   erase
   loescht den Bildschirm
   closes the postscript file, renames it
   and opens it again.
--------------------------------------------------------------------*/
void erase()
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern int x_gr_width, x_gr_height;

  if(x_gr_dpy)
  { XClearWindow(x_gr_dpy, x_gr_wd);
    XSetForeground(x_gr_dpy, x_gr_gc, x_gr_farbe[x_gr_color_b].wert);
    XFillRectangle(x_gr_dpy, x_gr_pixmap, x_gr_gc, 0, 0, x_gr_width, x_gr_height);
    XSetForeground(x_gr_dpy, x_gr_gc, x_gr_farbe[x_gr_color_f].wert);
  }
  if(x_gr_fp)
  {
    closeps(1);
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    get_color(modus)
    modus =-1:  setzt Farbtafel und gibt Anzahl der Farben zurueck
    modus = 0: Hintergrundfarbe
    modus = 1: Vordergrundfarbe
--------------------------------------------------------------------*/
int get_color(int modus)
{
  extern Display *x_gr_dpy;
  extern int x_gr_int_screen;
  extern  int x_gr_depth;
  extern int x_gr_color_f, x_gr_color_b;
  
  int i;
  Visual *visual;
  Colormap cmap;
  XColor exact_def;

  if(x_gr_dpy)
  {
    visual = DefaultVisual(x_gr_dpy, x_gr_int_screen);
    cmap = DefaultColormap(x_gr_dpy, x_gr_int_screen);
    if(modus > 0) return x_gr_color_f;
    if(modus == 0) return x_gr_color_b; 
    if(x_gr_depth == 1)
    {
      x_gr_farbe[0].wert = BlackPixel(x_gr_dpy, x_gr_int_screen);
      x_gr_farbe[1].wert = WhitePixel(x_gr_dpy, x_gr_int_screen);
      for(i = 2; i < (int) x_gr_MAX_COLORS; ++i)
        x_gr_farbe[i].wert = BlackPixel(x_gr_dpy, x_gr_int_screen);
    }
    else
    { for (i = 0; i < (int) x_gr_MAX_COLORS; ++i)
      {
        if(!XParseColor(x_gr_dpy, cmap, x_gr_farbe[i].name, &exact_def))
        { fprintf(stderr, "Grafik: Farbe %s nicht gefunden,",
          x_gr_farbe[i].name);
          printf(" wird durch schwarz ersetzt\n");
          exact_def.pixel = 0;
        }
        else
        {
          if(!XAllocColor(x_gr_dpy, cmap, &exact_def))
          { fprintf(stderr, "Grafik: all colorcells allocated and read/write\n");
          }
        }
        x_gr_farbe[i].wert = exact_def.pixel;
      }
    }
    return (x_gr_depth > 1 ? x_gr_MAX_COLORS : 1);
  }
  else return -1;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  procedure    int get_mouse_position
               gibt die Koordinaten der Mausposition zurueck

  Rueckgabewert: 0
--------------------------------------------------------------------*/
int get_mouse_position(double *x, double *y)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern double x_gr_offset_x, x_gr_offset_y, x_gr_slope_x, x_gr_slope_y;
  extern int x_gr_current_x, x_gr_current_y;
  Window root, child;
  Cursor cursor;
  int irx, iry;
  unsigned int key_button;

  if(x_gr_dpy)
  {
    cursor = XCreateFontCursor(x_gr_dpy, 34);
    XDefineCursor(x_gr_dpy, x_gr_wd, cursor);
    XFlush(x_gr_dpy);
    while(!x_get_Event(ButtonPress, 1));
    XQueryPointer(x_gr_dpy, x_gr_wd, &root, &child, &irx, &iry,
                      &x_gr_current_x, &x_gr_current_y, &key_button);
    *x = (x_gr_current_x - x_gr_offset_x) / x_gr_slope_x;
    *y = (x_gr_current_y - x_gr_offset_y) / x_gr_slope_y;
    XFreeCursor(x_gr_dpy, cursor);
    cursor = XCreateFontCursor(x_gr_dpy, 132);
    XDefineCursor(x_gr_dpy, x_gr_wd, cursor);
  }
  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    int x_get_Event(int event.type, int wait_on_Event)
    call Event handler and returns event Type
    or 0 if no event has been found
--------------------------------------------------------------------*/
int x_get_Event(int eventtype, int wait_on_Event)
{
  extern Display *x_gr_dpy;
  XEvent event;
  int flag;

  do
  {
/* wait for next event */
    if(wait_on_Event > 0)  XNextEvent(x_gr_dpy, &event);
    else
    {
/* check whether an event is queued,, read it or return 0 */
      if(XEventsQueued(x_gr_dpy, QueuedAfterFlush)) XNextEvent(x_gr_dpy, &event);
      else return 0;
    }
    if((flag = (event.type == eventtype))) wait_on_Event = 0;
/* serve standard-events */
    switch(event.type)
    {
       case KeyPress: break;
       case Expose:
                   XCopyArea(x_gr_dpy, x_gr_pixmap, x_gr_wd, x_gr_gc,
                        0, 0, x_gr_width, x_gr_height, 0, 0);
                   XFlush(x_gr_dpy); break;
          default: 	break;
    }
  }while(wait_on_Event);
  return flag;
}

 
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    get_f_scale(xmin, ymin, xmax, ymax)
     gibt die Randkoordinaten zurueck
--------------------------------------------------------------------*/
void get_f_scale(double *x_min, double *y_min, double *x_max, double *y_max)
{
  extern double x_gr_offset_x, x_gr_offset_y, x_gr_slope_x, x_gr_slope_y;
  extern Display *x_gr_dpy;

  if(x_gr_dpy)
  {
    *x_min = -x_gr_offset_x / x_gr_slope_x;
    *x_max = x_gr_slope_x * x_gr_xwa.width + *x_min; 
    *y_min = -x_gr_offset_y / x_gr_slope_x;
    *y_max = x_gr_slope_y * x_gr_xwa.height + *y_min; 
  }
  else *x_min = *y_min = *x_max = *y_max = 0.;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  f_arc(x, y, rx, ry,  wa, we)
  zieht einen Kreisbogen um (x, y) mit Radien rx und ry von Winkel wa zu we
--------------------------------------------------------------------*/
void f_arc(double x, double y, double rx, double ry, double wa, double we)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;
  int wa64, dw64;
  double dw;
  unsigned int width, height;

  dw = we - wa;
  while(dw < 0.) dw += 360.;
  if(dw >  360.) dw = 360.;
  while(wa < -360) wa += 360.;
  while(wa >  360) wa -= 360.;
  we = wa + dw;
  wa64 = 64 * wa;  dw64 = 64 * dw;
  height = (unsigned int) fabs(ry * x_gr_slope_y);
  width = (unsigned int) fabs(rx * x_gr_slope_x);
  x_gr_current_x = x_gr_coord_x(x);
  x_gr_current_y = x_gr_coord_y(y);

  if(x_gr_dpy)
  { XDrawArc(x_gr_dpy, x_gr_wd, x_gr_gc, x_gr_current_x - width, 
            x_gr_current_y - height, 2*width, 2*height, wa64, dw64);
    XDrawArc(x_gr_dpy, x_gr_pixmap, x_gr_gc, x_gr_current_x - width, 
            x_gr_current_y - height, 2*width, 2*height, wa64, dw64);
  }

  if(x_gr_fp)
  { fprintf(x_gr_fp, "stroke\nnewpath\n %.2f %.2f %.2f %.2f %.2f %.2f ellipse\n",
     x_fp_coord_x(x), x_fp_coord_y(y), fabs(rx*x_fp_slope_x), 
          fabs(ry*x_fp_slope_y), wa, we);
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  f_arc0(x, y, r, wa, we)
  zieht einen Kreisbogen um (x, y) mit Radius r von Winkel wa zu we
--------------------------------------------------------------------*/
void f_arc0(double x, double y, double r, double wa, double we)
{
  f_arc(x, y, r, r, wa, we);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  f_arc1(x, y, xa, ya, w)
  zieht einen Kreisbogen um (x, y) von (xa, ya) mit dem Winkel w
--------------------------------------------------------------------*/
void f_arc1(double x, double y, double xa, double ya, double w)

{
  double wa;

  wa = winkel_zu_hori(x, y, xa, ya) * 180.0 / M_PI;
  f_arc0(x, y, hypot(xa - x, ya - y), wa, wa + w);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_arc2(x, y, xa, ya, xe, ye)
   zieht einen Ellipsenbogen um (x, y) von (xa, ya) nach (xe, ye)
   Die Hauptachse der Ellipse liegen in den beiden Achsenrichtungen
--------------------------------------------------------------------*/
void f_arc2(double x, double y, double xa, double ya, double xe, double ye)
{ 
  double wa, we, a, b, tmp;

  wa = winkel_zu_hori(x, y, xa, ya) * 180.0 / M_PI;
  we = winkel_zu_hori(x, y, xe, ye) * 180.0 / M_PI;
  if(xa == xe || ya == ye)
    a = b = 0.5 * (hypot(xa - x, ya - y) + hypot(xe - x, ye - y));
  else
  {
    a = (ye-y)*(ye-y);         
    b = (xa-x)*(xa-x);
    tmp = a * b - (ya-y)*(ya-y)*(xe-x)*(xe-x);
    a = fabs(tmp / (a - (ya-y)*(ya-y)));
    b = fabs(tmp / (b - (xe-x)*(xe-x)));
    a = a > 0. ? sqrt(a) : 0.;
    b = b > 0. ? sqrt(b) : 0.;
  }
  f_arc(x, y, a, b, wa, we);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  f_arc_fill(x, y, rx, ry, wa, we)
  zieht einen Kreisbogen um (x, y) mit Radius r von Winkel wa zu we
--------------------------------------------------------------------*/
void f_arc_fill(double x, double y, double rx, double ry, double wa, double we)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;
  int wa64, we64;

  unsigned int width, height;

  while(we < wa) we += 360.;
  while(we > 720 + wa) we -= 360.;
  wa64 = 64 * wa;   we64 = 64 * we;
  height = (unsigned int) fabs(ry * x_gr_slope_y);
  width = (unsigned int) fabs(rx * x_gr_slope_x);
  x_gr_current_x = x_gr_coord_x(x);
  x_gr_current_y = x_gr_coord_y(y);

  if(x_gr_dpy)
  { XFillArc(x_gr_dpy, x_gr_wd, x_gr_gc, x_gr_current_x - width, 
            x_gr_current_y - height, 2*width, 2*height, wa64, we64);
    XFillArc(x_gr_dpy, x_gr_pixmap, x_gr_gc, x_gr_current_x - width, 
            x_gr_current_y - height, 2*width, 2*height, wa64, we64);
  }

  if(x_gr_fp)
  { fprintf(x_gr_fp, "stroke\nnewpath\n %.2f %.2f %.2f %.2f %.2f %.2f ellipse\nfill\n",
     x_fp_coord_x(x), x_fp_coord_y(y), fabs(rx*x_fp_slope_x),
         fabs(ry*x_fp_slope_y), wa, we);
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  f_arrow(x0, y0, x1, y1, arrowlength)
  zeichnet eine Linie zwischen den spezifizierten Punkten
  mit einem Pfeil am Endpunkt
  Achtung: die Laenge des Pfeiles ist in Pixeleinheiten!
--------------------------------------------------------------------*/
void f_arrow(double x0, double yy0, double x1, double yy1, double length)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;
  double phi;
  XPoint points[3];

  points[0].x = x_gr_current_x = x_gr_coord_x(x1);
  points[0].y = x_gr_current_y = x_gr_coord_y(yy1);

  if(hypot(x0 - x1, yy0 - yy1) == 0)
  {
    f_point(x0, yy0);
    return;
  }
  if(x_gr_dpy)
  { int ix, iy;
    ix = x_gr_coord_x(x0);      iy = x_gr_coord_y(yy0);
    XDrawLine(x_gr_dpy, x_gr_wd,x_gr_gc, ix, iy,
                                        x_gr_current_x, x_gr_current_y);
    XDrawLine(x_gr_dpy, x_gr_pixmap, x_gr_gc, ix, iy,
                                        x_gr_current_x, x_gr_current_y);
    ix = x_gr_current_x - ix;
    iy = x_gr_current_y - iy;
    if(hypot((double) ix, (double) iy) > 0.)
    {
      phi = atan2((double) iy,(double) ix);
      points[1].x = x_gr_current_x - length * cos(phi - 0.3);
      points[1].y = x_gr_current_y - length * sin(phi - 0.3);
      points[2].x = x_gr_current_x - length * cos(phi + 0.3);
      points[2].y = x_gr_current_y - length * sin(phi + 0.3);
      XFillPolygon(x_gr_dpy, x_gr_wd, x_gr_gc, points, 3, Convex, CoordModeOrigin);
      XFillPolygon(x_gr_dpy, x_gr_pixmap, x_gr_gc, points, 3, Convex, CoordModeOrigin);
    }
  }
  if(x_gr_fp)
  {
    double ix, iy, ix1, iy1, ix2, iy2;
    double width, height;
    ix = x_fp_coord_x(x0);
    iy = x_fp_coord_y(yy0);
    width =  (x1 - x0) * x_fp_slope_x;
    height = (yy1 - yy0) * x_fp_slope_y;
    if(++x_gr_count > 100)
    {
      x_gr_count = 0;
      fprintf(x_gr_fp, "stroke\n");
    }
    fprintf(x_gr_fp, " %.2f %.2f moveto\n %.2f %.2f rlineto\n",
                       ix, iy, width, height);
    ix += width;   iy += height;
    if(hypot(width, height) > 0.)
    {
      phi = atan2(height, width);
      length *= 0.375;
      ix1 = ix - length * cos(phi - 0.3);
      iy1 = iy - length * sin(phi - 0.3);
      ix2 = ix - length * cos(phi + 0.3);
      iy2 = iy - length * sin(phi + 0.3);
      fprintf(x_gr_fp, "stroke\nnewpath\n %.2f %.2f moveto\n", ix, iy);
      fprintf(x_gr_fp, " %.2f %.2f rlineto\n", ix1 - ix, iy1 -iy);
      fprintf(x_gr_fp, " %.2f %.2f rlineto\n closepath fill\n",
                        ix2 - ix1, iy2 - iy1);
    }
  }
}



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  f_box(x0, y0, x1, y1)
   zeichnet einen Rahmen mit den spezifizierten Eckpunkten
--------------------------------------------------------------------*/
void f_box(double xa, double ya, double xe, double ye)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;

  x_gr_current_x = x_gr_coord_x(xa);
  x_gr_current_y = x_gr_coord_y(ya);
  if(x_gr_dpy)
  { 
    int ix, iy;
    unsigned int width, height;
    ix = x_gr_coord_x(xe > xa ? xa : xe);
    iy = x_gr_coord_y(ye > ya ? ye : ya);
    width =  (unsigned int) fabs((xa - xe) * x_gr_slope_x);
    height = (unsigned int) fabs((ya - ye) * x_gr_slope_y);

    XDrawRectangle(x_gr_dpy, x_gr_wd, x_gr_gc, ix, iy, width, height);
    XDrawRectangle(x_gr_dpy, x_gr_pixmap, x_gr_gc, ix, iy, width, height);
  }
  if(x_gr_fp)
  {
    double ix, iy;
    double width, height;
    ix = x_fp_coord_x(xe > xa ? xa : xe);
    iy = x_fp_coord_y(ye > ya ? ya : ye);
    width =  fabs((xa - xe) * x_fp_slope_x);
    height = fabs((ya - ye) * x_fp_slope_y);
    fprintf(x_gr_fp, "stroke\nnewpath\n %.2f %.2f moveto\n  0. %.2f rlineto\n",
                       ix, iy, height);
    fprintf(x_gr_fp, " %.2f 0 rlineto\n  0 %.2f rlineto\n closepath\n",
                       width, -height);
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_box_fill(xa, ya, xe, ye)
   Zeichnet ein gefuelltes Rechteck zwischen den spezifizierten Punkten
--------------------------------------------------------------------*/
void f_box_fill(double xa, double ya, double xe, double ye)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;

  x_gr_current_x = x_gr_coord_x(xa);
  x_gr_current_y = x_gr_coord_y(ya);
  if(x_gr_dpy)
  { 
    int ix, iy;
    unsigned int width, height;
    ix = x_gr_coord_x(xe > xa ? xa : xe);
    iy = x_gr_coord_y(ye > ya ? ye : ya);
    width =  (unsigned int) fabs((xa - xe) * x_gr_slope_x);
    height = (unsigned int) fabs((ya - ye) * x_gr_slope_y);
    XFillRectangle(x_gr_dpy, x_gr_wd, x_gr_gc, ix, iy, width, height);
    XFillRectangle(x_gr_dpy, x_gr_pixmap, x_gr_gc, ix, iy, width, height);
  }
  if(x_gr_fp)
  {
    double ix, iy;
    double width, height;
    ix = x_fp_coord_x(xe > xa ? xa : xe);
    iy = x_fp_coord_y(ye > ya ? ya : ye);
    width =  fabs((xa - xe) * x_fp_slope_x);
    height = fabs((ya - ye) * x_fp_slope_y);
    fprintf(x_gr_fp, "stroke\nnewpath\n %.2f %.2f moveto\n  0. %.2f rlineto\n",
                       ix, iy, height);
    fprintf(x_gr_fp, " %.2f 0 rlineto\n  0 %.2f rlineto\n closepath fill\n",
                       width, -height);
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_circle(x, y, r)
   zeichnet einen Kreis mit Radius r um (x, y)
--------------------------------------------------------------------*/
void f_circle(double x, double y, double r)
{
  extern double x_gr_slope_x, x_gr_slope_y;
  f_arc(x, y, r, r * x_gr_slope_x / x_gr_slope_y, 0., 360.);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_circle_fill(x, y, r)
   zeichnet einen gefuellten Kreis mit Radius r um (x, y)
--------------------------------------------------------------------*/
void f_circle_fill(double x, double y, double r)
{
  extern double x_gr_slope_x, x_gr_slope_y;
  f_arc_fill(x, y, r, r * x_gr_slope_x / x_gr_slope_y, 0., 360.);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  f_cont(x, y)
  zeichnet eine Linie von den alten Koordinaten zu (x, y)
--------------------------------------------------------------------*/
void f_cont(double x, double y)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;
  extern FILE *x_gr_fp;
  int ix, iy;

  ix = x_gr_coord_x(x);
  iy = x_gr_coord_y(y);

  if(x_gr_dpy)
  { XDrawLine(x_gr_dpy, x_gr_wd, x_gr_gc, x_gr_current_x, x_gr_current_y,
                                         ix, iy);
    XDrawLine(x_gr_dpy, x_gr_pixmap, x_gr_gc,  x_gr_current_x, x_gr_current_y,
                                          ix, iy);
  }
  if(x_gr_fp)
  {
    fprintf(x_gr_fp, " %.2f %.2f lineto\n", x_fp_coord_x(x), x_fp_coord_y(y));
    ++x_gr_count;
  }
  x_gr_current_x = ix;
  x_gr_current_y = iy; 
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_ellipse(x, y, rx, ry)
   zeichnet eine Ellipse mit Radien rx und ry um (x, y)
--------------------------------------------------------------------*/
void f_ellipse(double x, double y, double rx, double ry)
{
  f_arc(x, y, rx, ry, 0., 360.);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_ellipse_fill(x, y, rx, ry)
   zeichnet eine gefuellteEllipse mit Radien rx und ry um (x, y)
--------------------------------------------------------------------*/
void f_ellipse_fill(double x, double y, double rx, double ry)
{
  f_arc_fill(x, y, rx, ry, 0., 360.);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  f_line(x0, y0, x1, y1)
  zeichnet eine Linie zwischen den spezifizierten Punkten
--------------------------------------------------------------------*/
void f_line(double x0, double yy0, double x1, double yy1)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;

  x_gr_current_x = x_gr_coord_x(x1);
  x_gr_current_y = x_gr_coord_y(yy1);

  if(x_gr_dpy)
  { int ix, iy;
    ix = x_gr_coord_x(x0);      iy = x_gr_coord_y(yy0);
    XDrawLine(x_gr_dpy, x_gr_wd,x_gr_gc, ix, iy,
                                        x_gr_current_x, x_gr_current_y);
    XDrawLine(x_gr_dpy, x_gr_pixmap, x_gr_gc, ix, iy,
                                        x_gr_current_x, x_gr_current_y);
  }
 
  if(x_gr_fp)
  {
    double ix, iy;
    double width, height;
    ix = x_fp_coord_x(x0);
    iy = x_fp_coord_y(yy0);
    width =  (x1 - x0) * x_fp_slope_x;
    height = (yy1 - yy0) * x_fp_slope_y;
    if(++x_gr_count > 100)
    {
      x_gr_count = 0;
      fprintf(x_gr_fp, "stroke\n");
    }
    fprintf(x_gr_fp, " %.2f %.2f moveto\n %.2f %.2f rlineto\n",
                       ix, iy, width, height);
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_move(x, y)
   setzt die Cursor Position auf (x, y)
--------------------------------------------------------------------*/
void f_move(double x, double y)

{
  extern int x_gr_current_x, x_gr_current_y;
  extern FILE *x_gr_fp;

  x_gr_current_x = x_gr_coord_x(x);
  x_gr_current_y = x_gr_coord_y(y);
  if(x_gr_fp)
  {
    if(++x_gr_count > 100)
    {
      x_gr_count = 0;
      fprintf(x_gr_fp, "stroke\n");
    }
    fprintf(x_gr_fp, " %.2f %.2f moveto\n", x_fp_coord_x(x), x_fp_coord_y(y));
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     f_point(x, y)
     zeichnet bei der angegeben Koordinate einen Punkt
--------------------------------------------------------------------*/
void f_point(double x, double y)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;
  extern FILE *x_gr_fp;

  x_gr_current_x = x_gr_coord_x(x);
  x_gr_current_y = x_gr_coord_y(y);

  if(x_gr_dpy)
  {
    XDrawPoint(x_gr_dpy, x_gr_wd, x_gr_gc, x_gr_current_x, x_gr_current_y);
    XDrawPoint(x_gr_dpy, x_gr_pixmap, x_gr_gc, x_gr_current_x, x_gr_current_y);
  }
  if(x_gr_fp)
  {
    fprintf(x_gr_fp, " %.2f %.2f moveto\n 1 0 rlineto\n",
                   x_fp_coord_x(x), x_fp_coord_y(y));
    if(++x_gr_count > 100000)
    {
      x_gr_count = 0;
      fprintf(x_gr_fp, "stroke\n");
    }
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_poly(*x, *y, npoints)
   zeichnet ein Polygon mit npoints Punkte x[i], y[i]
--------------------------------------------------------------------*/
void f_poly(double *x, double *y, int npoints)
{
  int i;
  f_move(x[0], y[0]);
  for(i = 1; i < npoints; ++i) f_cont(x[i], y[i]);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_poly_fill(*x, *y, npoints)
   zeichnet ein gefuelltes Polygon mit npoints Punkte x[i], y[i]
--------------------------------------------------------------------*/
void f_poly_fill(double *x, double *y, int npoints)
{
  int i;
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;

  x_gr_current_x = x_gr_coord_x(*x);
  x_gr_current_y = x_gr_coord_y(*y);
  if(x_gr_dpy)
  { 
    XPoint *points;
    if((points = (XPoint *) calloc(npoints, sizeof(XPoint))))
    {
      for(i = 0; i < npoints; ++i)
      {
        points[i].x = x_gr_coord_x(x[i]);
        points[i].y = x_gr_coord_y(y[i]);
      }
      XFillPolygon(x_gr_dpy, x_gr_wd, x_gr_gc, points, npoints, Complex, CoordModeOrigin);
      XFillPolygon(x_gr_dpy, x_gr_pixmap, x_gr_gc, points, npoints, Complex, CoordModeOrigin);
      free(points);
    }
  }
  if(x_gr_fp)
  {
    double ix, iy;
    ix = x_fp_coord_x(x[0]);
    iy = x_fp_coord_y(y[0]);
    fprintf(x_gr_fp, "stroke\nnewpath\n %.2f %.2f moveto\n", ix, iy);
    for(i = 1; i < npoints; ++i)
    {
      ix = x_fp_coord_x(x[i]);
      iy = x_fp_coord_y(y[i]);
      fprintf(x_gr_fp, " %.2f %.2f lineto\n", ix, iy);
    }
    fprintf(x_gr_fp, "closepath fill\n");
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   f_space(xmin, ymin, xmax, ymax)
   siehe set_f_scale
--------------------------------------------------------------------*/
void f_space(double x_min, double y_min, double x_max, double y_max)
{
  set_f_scale(x_min, y_min, x_max, y_max);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   flushpl()
   leert den Zwischenpuffer
--------------------------------------------------------------------*/
void flushpl(void)
{
  extern Display *x_gr_dpy;
  if(x_gr_dpy) XFlush(x_gr_dpy);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    function  kbhit()
              ueberprueft, ob eingabe im Grafikfenster gemacht wurde
              gibt null zurueck falls nein,
              gibt das Zeichen zurueck falls ja
              entspricht inetwa der MSDOS-Routine
--------------------------------------------------------------------*/
int kbhit(void)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;

  XEvent event;
  KeySym keysym;
  XComposeStatus status;
  char string[10];

  if(x_gr_dpy)
  {
    if(XCheckWindowEvent(x_gr_dpy, x_gr_wd, KeyPressMask, &event))
    { *string = '\0';
      (void) XLookupString(&(event.xkey), string, 4, &keysym, &status);
      return ((int) *string);
    }
  }
  return 0;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  label(string, groesse)
  schreibt beginnend an der momentanen Cursorposition einen String
--------------------------------------------------------------------*/
void label(char *s, int size)
{
  extern Display *x_gr_dpy;
  extern Window x_gr_wd;
  extern Pixmap x_gr_pixmap;
  extern GC x_gr_gc;
  extern int x_gr_current_x, x_gr_current_y;
  extern char *x_gr_font_names[];
  extern int x_gr_fontsize;
  
  x_gr_fontsize = size < 0 ? 0 : (size <=3 ? size : 3);
  if(x_gr_dpy)
  { 
    if(x_gr_old_fontsize != x_gr_fontsize)
    { x_gr_old_fontsize = x_gr_fontsize;
      if(load_font(x_gr_font_names[x_gr_fontsize]))
        if(load_font(x_gr_font_names[x_gr_fontsize+4]))        
          fprintf(stderr, " Cannot load font %s\n", x_gr_font_names[x_gr_fontsize+4]);
    }
    XDrawString(x_gr_dpy, x_gr_wd, x_gr_gc, x_gr_current_x, x_gr_current_y,
            s, strlen(s));
    XDrawString(x_gr_dpy, x_gr_pixmap, x_gr_gc, x_gr_current_x, x_gr_current_y,
            s, strlen(s));
  }
  if(x_gr_fp && *s)
  { 
    if(x_fp_old_fontsize != x_gr_fontsize)
    { x_fp_old_fontsize = x_gr_fontsize;
      fprintf(x_gr_fp, "/Times-Roman findfont\n%d scalefont\nsetfont\n",
                         x_fp_fontsize[x_gr_fontsize]);
    }
    if(*s) fprintf(x_gr_fp, "(%s) show\n", s);
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
--------------------------------------------------------------------*/
void labelp(char *string, int size)     /* Fuer Pascal */
{
  label(string, size);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
--------------------------------------------------------------------*/
void label1(x,l,n,e)
float x;
unsigned l,n,e;
{
  static char format[]="feEgG", form[]="f";
  char steuer[12], le[4], nk[4];
  char string[31];


  l=(l>25)? 25:l; n=(n>7)? 7:n;
  e=(e>9)? 0:e;
  form[0]=format[(e%5)];
  sprintf(le,"%d",l);
  sprintf(nk,"%d",n);
  strcpy(steuer,"%");
  if(e>=5) strcat(steuer," ");
  strcat(steuer,le);
  strcat(steuer,".");
  strcat(steuer,nk);
  strcat(steuer,form);
  sprintf(string,steuer,x);

  label(string, 2);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   linemod(string)
   setzt den Linienmodus
--------------------------------------------------------------------*/
void linemod(char *string)
{
  extern Display *x_gr_dpy;
  extern XGCValues xgcv;
  extern GC x_gr_gc;
  extern char *x_gr_line_styles[];
  int i,zeiger;
#ifdef HP
  static char dotted[]={1, 2};
  static char short_dashed[]={3, 3};
  static char long_dashed[]={6, 6}; 
  static char dot_dashed[]={1, 2, 4, 2};
  static char *dash_list[] =       { dotted,
                                     short_dashed,
                                     long_dashed,
                                     dot_dashed,
                                    };
  static unsigned char *fp_dash_list[] = {"",
                                          "1 3 1 3",
                                          "4 4 4 4",
                                          "8 8 8 8",
                                          "1 3 5 3",
					  };
#else
  static unsigned char dotted[]={1, 2};
  static unsigned char short_dashed[]={3, 3};
  static unsigned char long_dashed[]={6, 6}; 
  static unsigned char dot_dashed[]={1, 2, 4, 2};
  static unsigned char *dash_list[]={dotted,
                                     short_dashed,
                                     long_dashed,
                                     dot_dashed,
                                    };
  static  char *fp_dash_list[] = {"",
                                          "1 3 1 3",
                                          "4 4 4 4",
                                          "8 8 8 8",
                                          "1 3 5 3",
					  };
#endif
  static int dash_list_length[]={2, 2, 2, 4};
  static int dash_offset=0;
  for(i = 0, zeiger = -10; i < 5; i++)
    if(!strncmp(string,x_gr_line_styles[i],5)) zeiger = i;
  if(zeiger == -10)
  { if(!strncmp(string, "normal", 6)) zeiger = -1;
    else if(!strncmp(string, "bold", 4)) zeiger = -2;
  }
  if(x_gr_dpy)
  { if(zeiger == 0) 
      xgcv.line_style = LineSolid;
    else if(zeiger > 0)
    {
      xgcv.line_style=LineOnOffDash;
      XSetDashes(x_gr_dpy, x_gr_gc, dash_offset, dash_list[zeiger-1],
             dash_list_length[zeiger-1]);
    }
    else if(zeiger > -3)  xgcv.line_width = zeiger == -1 ? 1 : 2;
    if(zeiger > -3)
      XSetLineAttributes(x_gr_dpy, x_gr_gc, xgcv.line_width, xgcv.line_style, 
                   xgcv.cap_style, xgcv.join_style);
  }
  if(x_gr_fp)
  {
    if(zeiger >= 0) fprintf(x_gr_fp, "stroke\n [%s] 0 setdash\n", fp_dash_list[zeiger]);
    else if(zeiger > -3) fprintf(x_gr_fp, "stroke\n %.1f setlinewidth\n",
                                                       zeiger == -1 ? 0.5 : 1.0);
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
--------------------------------------------------------------------*/

void linemode(int modus)
{
  extern char *x_gr_line_styles[];
  char *name;
  
  switch(modus)
  {
    case NORMAL:       name = "normal";             break;
    case BOLD:         name = "bold";               break;  
    case DOTTED:       name = x_gr_line_styles[1];  break;
    case DOT_DASHED:   name = x_gr_line_styles[2];  break; 
    case SHORT_DASHED: name = x_gr_line_styles[3];  break; 
    case LONG_DASHED:  name = x_gr_line_styles[4];  break; 
    case SOLID:
    default:           name = x_gr_line_styles[0];  break;
  }
  linemod(name);
}

 
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
--------------------------------------------------------------------*/

int load_font(char *fontname)
{ 
  extern Display *x_gr_dpy;
  extern GC x_gr_gc;
  extern XFontStruct *x_gr_font_info;
  int flag = 1;

  if(x_gr_dpy)
  {
    if(x_gr_font_info) XFreeFont(x_gr_dpy, x_gr_font_info);
    x_gr_font_info = XLoadQueryFont(x_gr_dpy, fontname);
    if(x_gr_font_info)
    { XSetFont(x_gr_dpy, x_gr_gc, x_gr_font_info->fid);
      flag = 0;
    }
  }
  return flag;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  procedure    int select_color()
               zeigt eine Farbtafel und erlaubt mit der Maus die Auswahl
               einer Farbe. Gibt diese farbe zurueck.
--------------------------------------------------------------------*/
int select_color(void)
{
  extern Display *x_gr_dpy;
  extern double x_gr_offset_x, x_gr_offset_y, x_gr_slope_x, x_gr_slope_y;
  extern int x_gr_width, x_gr_height, x_gr_depth;
  extern int x_gr_color_f, x_gr_color_b;

  static char message[]={"Select color with mouse"};
  int i, maxcolor;
  double xm, ym, x0, yy0, x1, yy1, dy, x, y;
  
  if(!x_gr_dpy) return -1;
  maxcolor = x_gr_depth == 1 ? 2 : x_gr_MAX_COLORS;
  xm = -x_gr_offset_x / x_gr_slope_x;
  x1 = (x_gr_width - x_gr_offset_x) / x_gr_slope_x;
  i = 0.05 * x_gr_width < 20;
  i = i >= 20 ? i : (x_gr_width > 20 ? 20 : x_gr_width);
  x0 = x1 - i / x_gr_slope_x;
  yy1 = -x_gr_offset_y / x_gr_slope_y;
  y = yy0 = (x_gr_height - x_gr_offset_y) /x_gr_slope_y;
  dy = (yy1 - yy0) / maxcolor;
  xm = xm + 0.02 * (x1 - xm);
  ym = yy0 + 0.02 * (yy1 - yy0);
  set_color(x_gr_color_f);
  f_move(xm, ym);
  label(message, 2);
  for(i = 0; i < maxcolor; ++i)
  { set_color(i);
    yy1 = y + dy;
    f_box_fill(x0, y, x1, yy1);
    y = yy1;
  }
  do
  { get_mouse_position(&x, &y);
  }
  while(x < x0 || x > x1);
  i = (y - yy0) / dy;
  set_color(x_gr_color_b);
  f_box_fill(x0, yy0, x1, yy1);
  f_move(xm, ym);
  label(message, 2);
  set_color(x_gr_color_f);
  return i;
}
  

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   select_pen(pen)
   Routine ist nur zur Kompatibilitaet mit frueher Grafik da
--------------------------------------------------------------------*/
void select_pen(unsigned int pen)
{
  set_color(pen);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     set_color(int farbe)
     setzt die Vordergrundsfarbe
     fall Aufruf vor openpl() erfolgt, wird die Hintergrundsfarbe 
        gesetzt.
--------------------------------------------------------------------*/
void set_color(int farbe)
{
  extern Display *x_gr_dpy;
  extern FILE *x_gr_fp;
  extern GC x_gr_gc;
  extern int x_gr_color_b, x_gr_color_f;
  extern int x_gr_depth;

  if(x_gr_depth == 1) farbe = farbe % 2;
  else                farbe = farbe % x_gr_MAX_COLORS;
  if(x_gr_dpy)
  {
    x_gr_color_f = farbe; 
    XSetForeground(x_gr_dpy, x_gr_gc, x_gr_farbe[x_gr_color_f].wert);
  }
  else x_gr_color_b = farbe;
  if(x_gr_fp)
#ifdef PS_COLOR
    fprintf(x_gr_fp, "stroke\n%.4f %.4f %.4f setrgbcolor\n", 
      x_gr_fp_color[farbe].r/255., x_gr_fp_color[farbe].g/255., x_gr_fp_color[farbe].b/255.);
#else
    fprintf(x_gr_fp, "stroke\n%.2f setgray\n", x_gr_fp_color[farbe]);
#endif
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  procedure    set_f_scale
  related:     f_space
               get_f_scale
               x_gr_coord
               y_gr_coord
--------------------------------------------------------------------*/
void set_f_scale(double x_min, double y_min, double x_max, double y_max)
{
  extern Display *x_gr_dpy;
  extern double x_gr_offset_x, x_gr_offset_y, x_gr_slope_x, x_gr_slope_y;
  extern FILE *x_gr_fp;
  extern double x_fp_offset_x, x_fp_offset_y, x_fp_slope_x, x_fp_slope_y;
  extern double x_fp_width, x_fp_height;
  extern int x_gr_width, x_gr_height;

  if(x_min == x_max) x_max += 1.; 
  if(y_min == y_max) y_max += 1.; 
  x_gr_x_min = x_min;      x_gr_y_min = y_min;  
  x_gr_x_max = x_max;      x_gr_y_max = y_max;  

  if(x_gr_dpy)
  {
    x_gr_slope_x = x_gr_width / (x_max - x_min);
    x_gr_offset_x = -x_min * x_gr_slope_x;
    x_gr_slope_y = x_gr_height / (y_min - y_max);
    x_gr_offset_y = -y_max * x_gr_slope_y;
  }

  if(x_gr_fp)
  {
    x_fp_slope_x = x_fp_width / (x_max - x_min);
    x_fp_offset_x = FP_XMIN - x_min * x_fp_slope_x;
    x_fp_slope_y = x_fp_height / (y_max - y_min);
    x_fp_offset_y = FP_YMIN - y_min * x_fp_slope_y;
  }
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  procedure    set_window_size(xwidth, ywidth)
               setzt die Fenstergroesse in relativen Einheiten der
               Bildschirmkkoordinanten
               Aufruf muss vor openpl() erfolgen
--------------------------------------------------------------------*/
void set_window_size(double width, double height)
{ 
  extern double x_gr_rel_width, x_gr_rel_height;
  if(width  > 0. && width  <= 1.) x_gr_rel_width = width;
  if(height > 0. && height <= 1.) x_gr_rel_height = height;
}

