/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  int graph((double *) x, (double *) dx, (double *) y, (double *) dy, 
            (int)points, int(mode))

    Die Routine zeichnet Punktepaare
       (x[i] +- dx[i], y[i] +- dy[i])               ; 0 <= i <= points.
    in einen Graphen mit automatischer Koordinatenwahl.

    Der Parameter "mode" steuert den Bildaufbau:

    mode == 0: Der Display wird initialisiert, ein Rahmen wird mit den
                 Datenpunkten gezeichnet
    mode != 0: Der Display wird nur initialisiert, falls es noch nicht getan
                  wurde.
    mode < 0: log-Display fuer y-Achse

    Es brauchen nicht alle Datenpunkte angegeben zu werden:

    graph (x, 0, y, 0, points, mode):
	Die Punktepaare (x[i], y[i]) werden als Kreise gezeichnet (mode = 0)
		    werden mit einer linearen interpolation miteinander
                        verbunden (mode = 1)
		    werden mit einer Spline interpolation miteinander
                        verbunden (mode = 3).
        statt des Nullpointers kann dx durch x, dy durch y ersetzt werden.

    graph (x, 0, y, dy, points, mode):
	Die Punktepaare (x[i], y[i] +- dy[i]) werden als Kreise mit
                        Fehlerbalken in y-Richtung gezeichnet.

    graph (x, dx, y, dy, points, mode):
	Die Punktepaare (x[i] +- dx[i], y[i] +- dy[i]) werden als Kreise
                        mit Fehlerbalken in x- und y- Richtung gezeichnet,

    "0" ist hier der NULL - Pointer.

    Achsenbeschriftungen und einen Titel kann mit dem Aufruf
 
    graphlabel(char *xachsen_label, char *yachsenlabel, char *titel)

    erzeugt werden.

    Beschraenkung:
    Im aufrufenden Programm sollte an geeigneter Stelle die
       Graphikausgabe mit dem Befehl "closegr()" abgeschlossen werden.

    Author: Friedrich Riess, Juni 1988
            letzte Aenderung: 06.12.1991
-----------------------------------------------------------------------------*/
#include "grafik.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <malloc.h>

#define FAKX 0.05            /* relativer Abstand vom Rand, horizintale Richtung */
#define FAKY 0.05            /* dito, vertikale Richtung */
#define MARKX 0.015          /* Groesse der x - Marken */
#define MARKY 0.015          /* Groesse der y - Marken  */
#define XMARKDISTANZ 0.05    /* Minimum Abstand der x-Achsenwerte */
#define DOTSIZE 0.005        /* Groesse eines Kreises bei der Punkte Darstellung */
#define STEPGRAPH 0.002      /* Schrittweite bei der Spline Interpolation */
#define DNULL (double *) 0
#define SAVEFILE "graphplt.bak"

static int opengraph = 0, corealloc = 0, logdis = 0;
static double xmin, xmax, ymin, ymax;
static double xlaenge, ylaenge;
double *yidata = NULL, *dyidata = NULL;

double setze_ganzzahl(double*, double*);
int setze_markierung(double, double*);
void graphlabel(char *, char *, char *);
int spline_ini(double*, double*, double*, int);

int graph(double *xdata, double *dxdata, double *ydata, double *dydata,
          int points, int mode)
{
  static int color;
  double *pdxdata, *pdydata;
  double x, y, tmp, radius, bar;
  int i, j;
  char str[20];

/* copy y-array into internal buffer and take log of it if mode < 0 */
  if(corealloc)
  {
    if(yidata) free(yidata);
    if(dyidata) free(dyidata);
    yidata = dyidata = NULL;
    corealloc = 0;
  }
/* if memory request failed, use original arrays, no log display possible  */
  if((yidata = (double *) malloc(points*sizeof(double))) == NULL)
  {
    yidata = ydata;
    if(mode < 0) mode = -mode - 1;
  }
  else 
  {
    corealloc = 1;
    if(dydata && dydata != ydata)
    {
      if((dyidata = (double *) malloc(points*sizeof(double))) == NULL)
      {
        free(yidata);
        yidata = ydata;
        dyidata = dydata;
        corealloc = 0;
        if(mode < 0) mode = -mode - 1;
      }
    }
  }
  {
    logdis = 0;
    tmp = 1.e+100;
    for(i = 0; i < points; ++i)
    {
      if(mode >= 0)
      {
        yidata[i] = ydata[i];
        if(dyidata) dyidata[i] = dydata[i];
      }
      if(ydata[i] > 0. && tmp > ydata[i]) tmp = ydata[i];
    }
    if(mode < 0)
    {
      logdis = 1;
      mode = -mode - 1;
      tmp = log10(0.1*tmp);
      for(i = 0; i < points; ++i)
      {
        if(ydata[i] > 0.) yidata[i] = log10(ydata[i]);
        else              yidata[i] = tmp;
        if(dyidata && dyidata != dydata)
          dyidata[i] = dydata[i]/(M_E*(ydata[i] > 0. ? ydata[i] : 1.));
      }
    }
  }     
/* Bestimme Minium und Maximum der Werte */
  pdxdata = xdata == dxdata ? DNULL : dxdata;
  pdydata = ydata == dydata ? DNULL : dyidata;
  if(!opengraph)
  {
    int step;
    xmin = xmax = *xdata;      ymin = ymax = *yidata;
    for(i = 0; i < points; i++)
    { radius = pdxdata != DNULL ? fabs(dxdata[i]) : 0.;
      bar    = pdydata != DNULL ? fabs(dyidata[i]) : 0.;
      if(xmin > (tmp = xdata[i] - radius)) xmin = tmp;
      if(xmax < (tmp = xdata[i] + radius)) xmax = tmp;
      if(ymin > (tmp = yidata[i] - bar   )) ymin = tmp;
      if(ymax < (tmp = yidata[i] + bar   )) ymax = tmp;
    }
    if(xmin == xmax) { xmin -= 1.; xmax += 1.;}
    if(ymin == ymax) { ymin -= 1.; ymax += 1.;}
    if(logdis) { ymin = floor(ymin);  ymax = ceil(ymax);}
/* extrapoliere zu geeigneten ganzzahligen Werte */
    xlaenge = setze_ganzzahl(&xmin, &xmax);
    if(logdis) ylaenge = ymax - ymin;
    else       ylaenge = setze_ganzzahl(&ymin, &ymax);
/* eroeffne display und waehle Massstab */

    set_color(WEISS);
    set_window_size(0.6, 0.);
    if(openpf(SAVEFILE)) return -1;
    set_focus("root");
    erase();
    set_f_scale(xmin - 2 * FAKX * xlaenge, ymin -  2 * FAKY * ylaenge,
		xmax + FAKX * xlaenge, ymax + FAKY * ylaenge);
    color = get_color(1);
    linemod("bold");

/* Zeichne Rahmen   */
/* :   x - Achse     */
    tmp = XMARKDISTANZ;
    step = setze_markierung(xlaenge, &tmp);
    x = xmin; bar = MARKX * ylaenge;
    i = 0; if(xmin < 0. && xmax > 0.) i = (int) (-xmin / tmp + 0.01) % step;
    do
    { y = (y = x + tmp) > xmax ? xmax : y;
      if(i++ % step)
      { f_move(x, ymin + bar); f_cont(x, ymin);  f_cont(y, ymin);
      }
      else
      {
	if(xlaenge > 999999.) j = sprintf(str, "%.1e", x);
	else if(xlaenge > 20.) j = sprintf(str, "%.0f", x);
	     else if(xlaenge > 2.) j = sprintf(str, "%.1f", x);
		  else j = sprintf(str, "%.2f", x);
	f_move(x - 0.006 * j * xlaenge, ymin - 0.05 * ylaenge);
	label(str, 1);
        f_move(x, ymin + 2 * bar); f_cont(x, ymin); f_cont(y, ymin);
      }
      x = y;
    } while(x < xmax);
/* x-Achse oberer Teil */
    x = xmin;
    i = 0; if(xmin < 0. && xmax > 0.) i = (int)(-xmin / tmp + 0.01) % step;
    do
    { y = (y = x + tmp) > xmax ? xmax : y;
      if(i++ % step)
      { f_move(x, ymax - bar); f_cont(x, ymax);  f_cont(y, ymax);
      }
      else
      { f_move(x, ymax - 2 * bar); f_cont(x, ymax); f_cont(y, ymax);
      }
      x = y;
    } while(x < xmax);

/* :  y -Achse - links */
    tmp = XMARKDISTANZ;
    step = setze_markierung(ylaenge, &tmp);
    if(logdis) step = 1;
    y = ymin; bar = MARKY * xlaenge;
    i = 0; if(ymin < 0. && ymax > 0.) i = (int)(-ymin / tmp + 0.01) % step;
    do
    { x = (x = y + tmp) > ymax ? ymax : x;
      if(i++ % step)
      {
        if(!logdis)
	{
          f_move(xmin + bar, y); f_cont(xmin, y);  f_cont(xmin, x);
	}
      }
      else
      {
        if(logdis)
	{
          sprintf(str, "10^%.0f", y);
	}
          else
	{
	  if(ylaenge > 999999.) sprintf(str, "%7.2e", y);
	  else if(ylaenge > 10.) sprintf(str, "%7.0f", y);
	    else if(ylaenge > 1.) sprintf(str, "%7.1f", y);
	      else if(ylaenge > 0.1) sprintf(str,"%7.2f", y);
                else if(ylaenge > 0.01 || ylaenge < 1.e-5 * ylaenge) sprintf(str, "%7.3f", y);
		  else sprintf(str, "%7.2e", y);
	}
	f_move(xmin - 2 * FAKX * xlaenge, y - 0.005 * ylaenge);
	label(str, 1);
        if(logdis && y < ymax)
	{
          for(j = 2; j <= 9 && x < ymax; ++j)
	  {
            x = y + log10((double) j);
            f_move(xmin + bar, x); f_cont(xmin, x);
          }
          x = y + 1.;
	}
        f_move(xmin + 2 * bar, y); f_cont(xmin, y); f_cont(xmin, x);
      }
      y = x;
    } while(y < ymax);
/* :  y -Achse - rechts */
    y = ymin;
    i = 0; if(ymin < 0. && ymax > 0.) i = (int)(-ymin / tmp + 0.01) % step;
    do
    { x = (x = y + tmp) > ymax ? ymax : x;
      if(i++ % step && !logdis)
      { f_move(xmax - bar, y); f_cont(xmax, y);  f_cont(xmax, x);
      }
      else
      {
        if(logdis && y < ymax)
	{
          for(j = 2; j <= 9 && x < ymax; ++j)
	  {
            x = y + step*log10((double) j);
            f_move(xmax - bar, x); f_cont(xmax, x);
          }
          x = y + step;
	}
        f_move(xmax - 2 * bar, y); f_cont(xmax, y); f_cont(xmax, x);
      }
      y = x;
    } while(y < ymax);
/*  Null-Linien falls im Bild */
    if(xmin < 0. && xmax > 0.) f_line (0., ymin, 0., ymax);
    if(ymin < 0. && ymax > 0.) f_line (xmin, 0., xmax, 0.);
    opengraph = 1;
  }
  if(!mode)

/* Zeichne Daten-Punkte */
  { set_color(color);
    radius = DOTSIZE * xlaenge * (1. + 3 * FAKX);
    bar = DOTSIZE * ylaenge * (1. + 3 * FAKY);
    for(i = 0; i < points; i++)
    { f_circle(xdata[i], yidata[i], radius);
      if(pdxdata != DNULL)
      { if(dxdata[i] >= radius)
	{ x = xdata[i] - dxdata[i];
	  y = xdata[i] + dxdata[i];
	  f_line(x, yidata[i] - bar, x, yidata[i] + bar);
	  f_line(y, yidata[i] - bar, y, yidata[i] + bar);
	  f_line(x, yidata[i], y, yidata[i]);
	}
      }
      if(pdydata != DNULL)
      { if(dyidata[i] >= bar)
	{ y = yidata[i] - dyidata[i];
	  x = yidata[i] + dyidata[i];
	  f_line(xdata[i] - radius, y, xdata[i] + radius, y);
	  f_line(xdata[i] - radius, x, xdata[i] + radius, x);
	  f_line(xdata[i], y, xdata[i], x);
	}
      }
    }
    set_color(color);
  }
  else

/* Zeichne Fitdaten als Linie und macht eine kubische spline Interpolation
   falls der Abstand zwischen den Punkten zu gross ist (mode = 2) */
  {
    double step, a, b, *dy2;
    int flag;
    set_color(ROT);
    radius = STEPGRAPH * hypot(xlaenge, ylaenge);
    step = STEPGRAPH * xlaenge;
    dy2 = (double *) 0;
    for(i = 0, flag = 1; i < points; i++)
    {
      if(xdata[i] < xmin || xdata[i] > xmax || yidata[i] < ymin || yidata[i] > ymax)
      {
        mode = 1; flag = 1; continue;
      }
      if(flag)
      {
        flag = 0; f_move(xdata[i], yidata[i]); continue;
      }
      if(mode > 1)
      { if((tmp = hypot((bar = xdata[i] - xdata[i-1]), yidata[i] - yidata[i-1])) > radius)
	{ if(dy2 == (double *) 0)
	  { dy2 = (double *) calloc((unsigned)points, sizeof(double));
	    if(spline_ini(xdata, yidata, dy2, points)) mode = 1;
	  }
	  if(mode > 1)
	  { tmp = step * bar / tmp;
	    for(x = xdata[i-1] + tmp; x < xdata[i]; x += tmp)
	    {
	      a = (xdata[i] - x) / bar;
	      b = (x - xdata[i-1]) / bar;
	      y = a * yidata[i-1] + b * yidata[i]
		+ (a * (a * a - 1.) * dy2[i-1] + b * (b * b - 1.) * dy2[i]) * bar * bar / 6.;
	      f_cont(x, y);
	    }
	  }
	}
      }
      f_cont(xdata[i], yidata[i]);
    }
    set_color(color);
    if(dy2 != (double *) 0) free(dy2); 
  }
  return 0;
}

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    void graphlabel(char *xlabel, char *ylabel, char *titel)

    Diese Routine erlaubt das Beschriften der x-und y-achsee und das
    einsetzten eines Titels

    F. Riess, Dez. 1991
----------------------------------------------------------------------------*/
void graphlabel(char *xlabel, char *ylabel, char *titel)
{ int i;
  char tmp[2];
  if(xlabel != (char *) 0) 
  { i = strlen(xlabel);
    if(i > 50) i = 50;
    f_move(0.95*xmax-0.017*i*xlaenge, ymin-1.7*FAKY*ylaenge);
    label(xlabel, 2);
  }
  if(ylabel != (char *) 0)
  { tmp[1] = '\0';
    for(i = 0; ylabel[i] && i < 30; ++i)
    { f_move(xmin - 1.5 * FAKX * xlaenge, 0.95*ymax - 0.03 * i * ylaenge);
      *tmp = ylabel[i];
      label(tmp, 2);
    }
  }  
  if(titel != (char *) 0)
  { i = strlen(titel);
    if(i > 25) i =25;
    f_move(xmin+(0.5-0.017*i)*xlaenge, ymax + 0.4 * FAKY * ylaenge);
    label(titel, 2);
  }
  flushpl();
} 

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    double setze_ganzzahl((double *) min, (double *) max)

    Diese Routine nimmt zwei Grenzwerte min und max und extrapoliert sie
    in der Weise, dass das Resultat moeglichst nahe bei bei Zehnerpotenzen
    liegt.
    Zurueckgegeben werden die neuen Werte min und max und als Funktionswert
    die Differenz xmax - xmin.

    F. Riess, Juni 1988
----------------------------------------------------------------------------*/

double setze_ganzzahl(double *min, double *max)
{ double  save, tmp;
  tmp = (tmp = *max - *min) > 0. ? tmp : 1.;
  tmp = pow(10., floor(log10(tmp)));
  save = fabs(*min);
  *min = floor(*min / tmp) * tmp;
  if(save < 0.1 * fabs(*min)) *min *= 0.1;
  else if(save < 0.2 * fabs(*min)) *min *= 0.2;
       else if(save < 0.5 * fabs(*min)) *min *= 0.5;
  save = fabs(*max);
  *max =  ceil(*max / tmp) * tmp;
  if(save < 0.1 * fabs(*max)) *max *= 0.1;
  else if(save < 0.2 * fabs(*max)) *max *= 0.2;
       else if(save < 0.5 * fabs(*max)) *max *= 0.5;
  return (*max - *min);
}

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  int setze_markierung((double) achsen_laenge, (double *) schrittweite)

  Diese Routine teilt die achse mit der Laenge achsen_laenge in geeignete
  Abschnitte fuer Markierungen ein.
  Rueckgabewerte sind die Abstaende der Markierung in schrittweite und
  Rueckgabewert der Funktion sind die Anzahl der Schrittweiten, welche fuer eine
  Beschriftung gewaehlt werden koennen.

  F. Riess, Juni 1988
----------------------------------------------------------------------------*/

int setze_markierung(double achsen_laenge, double *schrittweite)
{ int step, marke;
  double potenz;
  if(*schrittweite == 0.) *schrittweite = 1.;
  *schrittweite *= achsen_laenge;
  potenz = floor(log10(*schrittweite));
  step = *schrittweite * pow(10., -potenz);
  marke = 5;
  if(step <= 1)
    step = 1;
  else
    if(step < 3)
      step = 2;
    else if(step < 7)
         { step = 5;
           marke = 4;
         }
         else step = 10;
  *schrittweite = step * pow(10., potenz);
  return marke;
}


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   int spline_ini(double *x, double *y, double *y2, int n)

   Initialisierungs routine fuer kubische splines: Berechnet die
   2. Ableitungen y2[i] fuer die Punkte Paare x[i], y[i] mit
   n Punkte.

   siehe: ?????? Numerical Recipies  Kapitel 3, Seite 88 ff
   Bei den Endpunkten wird eine "natuerliche" Ableitung angenommen.


----------------------------------------------------------------------------*/

int spline_ini(double *x, double *y, double *y2, int n)
{ double sig, p, *u;
  int i, okay = 1;
  u = (double *) calloc((unsigned) n, sizeof(double));
  for(i = 1, y2[0] = y2[n-1] = u[0] = u[n-1] = 0.; i < n-1; i++)
  {
    if(x[i-1] == x[i] || x[i] == x[i+1] || x[i-1] == x[i+1]) okay = 0;
    sig = (x[i] - x[i-1]) / (x[i+1] - x[i-1]);
    p = (p = sig * y2[i-1] + 2.) == 0. ? 1.e-10 : p;
    y2[i] = (sig - 1.) / p;
    u [i] = (6. * ((y[i+1] - y[i]) / (x[i+1] - x[i])
		 - (y[i] - y[i-1]) / (x[i] - x[i-1])) / (x[i+1] - x[i-1])
		 - sig * u[i-1]) / p;
  }
  if(okay) for(i = n-2; i >= 0; --i) y2[i] = y2[i] * y2[i+1] + u[i];
  free((char *)u);
  return !okay;
}

/*  closegr(mode)
    mode < 0 : close postscript file, do not close window
    mode = 0 : close window immediatly
    mode > 0 : close window interactive
*/
void closegr(int mode)
{
  if(opengraph)
  {
    flushpl();
    if(mode >= 0)
    {
      opengraph = 0;
      if(corealloc)
      {
        if(yidata) free(yidata);
        yidata = NULL;
        if(dyidata) free(dyidata);
        dyidata = NULL;
        corealloc = 0;
      }  
      closepl(mode ? 1 : 0);
    }      
    else closeps(1);
  }
}

/* Fortran Interface */

int graph_(double *xdata, double *dxdata, double *ydata, double *dydata,
           int *points, int *mode)
{  int i;
  i = graph(xdata, dxdata, ydata, dydata, *points, *mode);
  return i;
}

void graphlabel_(char *xlabel, char *ylabel, char *titel,
                 int xslabel, int yslabel,  int stitel)
{
  int i;
  char x[100], y[100], t[120];

  for(i = 0; i < xslabel; ++i) x[i] = xlabel[i];
  for(i = 0; i < yslabel; ++i) y[i] = ylabel[i];
  for(i = 0; i < stitel; ++i) t[i] = titel[i];
  x[xslabel] = y[yslabel] = t[stitel] = '\0';
  graphlabel(x, y, t);
}
  

void closegr_(int *mode)
{
  closegr(*mode);
}
